(**
   Implements a tab groupobject.

  TODO
  * Hide is suboptimal.
**)

MODULE VOTab;

(*
    Implements a tab gadget.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)


IMPORT D  := VODisplay,
       E  := VOEvent,
       G  := VOGUIObject,
       O  := VOObject;


CONST
  changedMsg * = 0;

  (* internal *)

  horizFrameThick = 2;
  vertFrameThick  = 2;

TYPE
  Prefs*     = POINTER TO PrefsDesc;

  (**
    In this class all preferences stuff of the button is stored.
  **)

  PrefsDesc* = RECORD (G.PrefsDesc)
                 hilightLabel* : BOOLEAN;
               END;

  TabEntry     = POINTER TO TabEntryDesc;
  TabEntryDesc = RECORD
                   next   : TabEntry;
                   label  : G.Object;
                   object : G.Object;
                 END;

  Tab*     = POINTER TO TabDesc;
  TabDesc* = RECORD (G.GroupDesc)
               prefs    : Prefs;

               mw       : LONGINT; (* greatest width of all tab-objects *)
               mh       : LONGINT; (* greatest height of all tab-object *)

               tabList,
               lastTab  : TabEntry;
               current  : G.Object;
               selected-: LONGINT; (* The currently selected tab (1..n) *)
               curCount : LONGINT;
               isIn     : BOOLEAN;
             END;

  (* messages *)

  ChangedMsg*     = POINTER TO ChangedMsgDesc;

  (**
    The PressedMsg generated everytime the button get clicked.
  **)

  ChangedMsgDesc* = RECORD (O.MessageDesc)
                      pos* : LONGINT;
                    END;

VAR
  prefs* : Prefs;


  PROCEDURE (p : Prefs) Init*;

  BEGIN
    p.Init^;

    p.hilightLabel:=FALSE;
  END Init;

  PROCEDURE (p : Prefs) SetPrefs(t : Tab);

  BEGIN
    t.prefs:=p;   (* We set the prefs *)

    IF p.background#NIL THEN
      t.SetBackgroundObject(p.background.Copy());
      t.backgroundObject.source:=t;
    END;
  END SetPrefs;

  PROCEDURE (t : Tab) Init*;

  BEGIN
    t.Init^;

    INCL(t.flags,G.canFocus);
    EXCL(t.flags,G.stdFocus);

    prefs.SetPrefs(t); (* We set the prefs *)

    t.tabList:=NIL;
    t.curCount:=0;
    t.selected:=1;
  END Init;

  PROCEDURE (t : Tab) Disable*(disable : BOOLEAN);

  BEGIN
    IF t.disabled#disable THEN
      t.disabled:=disable;
      t.Redraw;
    END;
  END Disable;

  PROCEDURE (t : Tab) AddTab*(label, object : G.Object);

  VAR
    entry : TabEntry;

  BEGIN
    NEW(entry);
    entry.label:=label;
    t.CopyBackground(entry.label);
    entry.object:=object;
    t.CopyBackground(entry.object);

    IF t.tabList=NIL THEN
      t.tabList:=entry;
    ELSE
      t.lastTab.next:=entry;
    END;
    t.lastTab:=entry;
    INC(t.count);
  END AddTab;

  PROCEDURE (t : Tab) CalcSize*(d : D.Display);

  VAR
    entry : TabEntry;
    ow,oh : LONGINT;

  BEGIN
    t.width:=0;
    t.height:=0;
    t.mw:=0;
    t.mh:=0;
    ow:=0;
    oh:=0;

    entry:=t.tabList;
    WHILE entry#NIL DO
      IF entry.label.StdFocus() THEN
        INCL(t.flags,G.stdFocus);
      END;
      entry.label.CalcSize(d);
      t.mw:=G.MaxLong(t.mw,entry.label.oWidth);
      t.mh:=G.MaxLong(t.mh,entry.label.oHeight);
      entry.object.CalcSize(d);
      ow:=G.MaxLong(ow,entry.object.oWidth);
      oh:=G.MaxLong(oh,entry.object.oHeight);
      entry:=entry.next;
    END;

    IF ~t.StdFocus() & t.MayFocus() THEN
      entry:=t.tabList;
      WHILE entry#NIL DO
        INCL(entry.label.flags,G.mayFocus);
        entry:=entry.next;
      END;
    END;

    INC(t.mh,d.spaceHeight+2*horizFrameThick);
    INC(t.mw,d.spaceWidth+2*vertFrameThick);
    INC(ow,2*d.spaceWidth);
    INC(oh,2*d.spaceHeight);

    t.height:=t.mh+oh+horizFrameThick;
    t.width:=G.MaxLong(ow+2*vertFrameThick,d.spaceWidth+t.count*t.mw+d.spaceWidth);

    t.minWidth:=t.width;
    t.minHeight:=t.height;

    t.CalcSize^(d);
  END CalcSize;

  PROCEDURE (t : Tab) GetEntry(count :LONGINT): TabEntry;

  VAR
    entry : TabEntry;

  BEGIN
    entry:=t.tabList;
    WHILE count>1 DO
      entry:=entry.next;
      DEC(count);
    END;
    RETURN entry;
  END GetEntry;

  PROCEDURE (t : Tab) DrawTab(count : LONGINT; selected : BOOLEAN);

  VAR
    x      : LONGINT;
    entry  : TabEntry;
    object : G.Object;

  BEGIN
    entry:=t.GetEntry(count);
    object:=entry.label;

    x:=t.x+t.draw.display.spaceWidth+(count-1)*t.mw;

    IF selected & t.prefs.hilightLabel THEN
      t.draw.mode:={D.selected};
    END;
    object.SetFlags({G.horizontalFlex,G.verticalFlex});
    object.Resize(t.mw-2*horizFrameThick,
                  t.mh-2*vertFrameThick);
    object.Draw(  x+(t.mw-object.oWidth) DIV 2,
                t.y+(t.mh-object.oHeight) DIV 2,t.draw);
    t.draw.mode:={};

    IF selected THEN
      t.draw.PushForeground(D.shadowColor);
    ELSE
      t.draw.PushForeground(D.shineColor);
    END;
    t.draw.DrawLine(x+3,t.y  ,x+t.mw-4,t.y);
    t.draw.DrawLine(x+1,t.y+1,x+t.mw-3,t.y+1);
    t.draw.DrawLine(x  ,t.y+3,x       ,t.y+t.mh-1-horizFrameThick);
    t.draw.DrawLine(x+1,t.y+1 ,x+1     ,t.y+t.mh-1-horizFrameThick);
    t.draw.PopForeground;

    IF selected THEN
      t.draw.PushForeground(D.shineColor);
    ELSE
      t.draw.PushForeground(D.shadowColor);
    END;
    t.draw.DrawLine(x+t.mw-1,t.y+3,x+t.mw-1,t.y+t.mh-1-horizFrameThick);
    t.draw.DrawLine(x+t.mw-2,t.y+1,x+t.mw-2,t.y+t.mh-1-horizFrameThick);
    t.draw.PopForeground;
  END DrawTab;

  PROCEDURE (t : Tab) DrawTop;

  BEGIN
    t.draw.PushForeground(D.shineColor);
    t.draw.DrawLine(t.x,t.y+t.mh-2,
                    t.x+(t.selected-1)*t.mw+t.draw.display.spaceWidth+1,t.y+t.mh-2);
    t.draw.DrawLine(t.x,t.y+t.mh-1,
                    t.x+(t.selected-1)*t.mw+t.draw.display.spaceWidth,t.y+t.mh-1);

    t.draw.DrawLine(t.x+(t.selected)*t.mw+t.draw.display.spaceWidth-vertFrameThick,t.y+t.mh-2,t.x+t.width-1,t.y+t.mh-2);
    t.draw.DrawLine(t.x+(t.selected)*t.mw+t.draw.display.spaceWidth-vertFrameThick,t.y+t.mh-1,t.x+t.width-2,t.y+t.mh-1);
    t.draw.PopForeground;

    t.draw.PushForeground(D.backgroundColor);
    t.draw.DrawLine(t.x+(t.selected-1)*t.mw+t.draw.display.spaceWidth+2             ,t.y+t.mh-2,
                    t.x+(t.selected)*t.mw+t.draw.display.spaceWidth-vertFrameThick-1,t.y+t.mh-2);
    t.draw.DrawLine(t.x+(t.selected-1)*t.mw+t.draw.display.spaceWidth+1             ,t.y+t.mh-1,
                    t.x+(t.selected)*t.mw+t.draw.display.spaceWidth-vertFrameThick-1,t.y+t.mh-1);
    t.draw.PopForeground;

    IF t.disabled THEN
      t.draw.PushForeground(D.disabledColor);
      t.draw.PushPattern(D.disablePattern,D.disableWidth,D.disableHeight,D.fgPattern);
      t.draw.FillRectangle(t.x,t.y,t.width,t.mh-horizFrameThick);
      t.draw.PopPattern;
      t.draw.PopForeground;
    END;
  END DrawTop;

  PROCEDURE (t : Tab) DrawObject(count : LONGINT);

  VAR
    entry : TabEntry;

  BEGIN
    entry:=t.GetEntry(count);
    IF (t.current#entry.object) & (t.current#NIL) THEN
      t.current.Hide;
    END;

    t.DrawBackground(t.x+2,t.y+t.mh,t.width-4,t.height-t.mh-2);

    t.current:=entry.object;
    t.current.Resize(t.width-2*vertFrameThick-2*t.draw.display.spaceWidth,
                     t.height-t.mh-horizFrameThick-2*t.draw.display.spaceWidth);
    t.current.Draw(t.x+(t.width-t.current.width) DIV 2,
                   t.y+t.mh+(t.height-t.mh-t.current.height) DIV 2,t.draw);
   t.selected:=count;
  END DrawObject;

  PROCEDURE (t : Tab) GetFocus*(event : E.Event):G.Object;

  VAR
    x : LONGINT;

  BEGIN
    IF ~t.visible OR t.disabled THEN
      RETURN NIL;
    END;

    WITH event : E.MouseEvent DO
      IF (event.type=E.mouseDown) & t.PointIsIn(event.x,event.y)
      & (event.qualifier={}) & (event.button=E.button1) THEN
        x:=t.x+t.draw.display.spaceWidth;
        IF (event.x>=x) & (event.x<=x+t.count*t.mw-1) & (event.y<=t.y+t.mh-1) THEN
          t.curCount:=(event.x-x) DIV t.mw +1;
          IF t.curCount#t.selected THEN
            t.DrawTab(t.curCount,TRUE);
            t.isIn:=TRUE;
            RETURN t;
          END;
        END;
      END;
    ELSE
    END;

    IF t.current#NIL THEN
      RETURN t.current.GetFocus(event);
    ELSE
      RETURN NIL;
    END;
  END GetFocus;

  PROCEDURE (t : Tab) HandleEvent*(event : E.Event):BOOLEAN;

  VAR
    x       : LONGINT;
    changed : ChangedMsg;

  BEGIN
    x:=t.x+t.draw.display.spaceWidth+(t.curCount-1)*t.mw;

    WITH event : E.MouseEvent DO
      IF (event.type=E.mouseUp) & (event.button=E.button1) THEN
        IF (event.x>=x) & (event.x<=x+t.mw-1) & (event.y>=t.y) & (event.y<=t.y+t.mh-1) THEN
          t.HideFocus;
          t.DrawTab(t.curCount,FALSE);
          t.DrawObject(t.curCount);
          t.DrawTop;
          t.DrawFocus;

          NEW(changed);
          changed.pos:=t.selected;
          t.Send(changed,changedMsg);
        END;
        RETURN TRUE;
      END;
    | event : E.MotionEvent DO
      IF (event.x>=x) & (event.x<=x+t.mw-1) & (event.y>=t.y) & (event.y<=t.y+t.mh-1) THEN
        IF ~t.isIn THEN
          t.DrawTab(t.curCount,TRUE);
          t.isIn:=TRUE;
        END;
      ELSIF t.isIn THEN
        t.DrawTab(t.curCount,FALSE);
        t.isIn:=FALSE;
      END;
    ELSE
    END;
    RETURN FALSE;
  END HandleEvent;

  PROCEDURE (t : Tab) HandleFocusEvent*(event : E.KeyEvent):BOOLEAN;

  VAR
    keysym : LONGINT;

  BEGIN
    IF event.type=E.keyDown THEN
      keysym:=event.GetKey();
      IF keysym=E.left THEN
        IF t.count>1 THEN
          t.HideFocus;
          DEC(t.selected);
          IF t.selected<1 THEN
            t.selected:=t.count;
          END;
          t.DrawObject(t.selected);
          t.DrawTop;
          t.DrawFocus;
        END;
        RETURN TRUE;
      ELSIF keysym=E.right THEN
        IF t.count>1 THEN
          t.HideFocus;
          INC(t.selected);
          IF t.selected>t.count THEN
            t.selected:=1;
          END;
          t.DrawObject(t.selected);
          t.DrawTop;
          t.DrawFocus;
        END;
        RETURN TRUE;
      END;
    END;
    RETURN FALSE;
  END HandleFocusEvent;

  PROCEDURE (t : Tab) GetPosObject*(x,y : LONGINT; type : LONGINT):G.Object;

  VAR
    object : TabEntry;
    return : G.Object;

  BEGIN
    IF t.current=NIL THEN
      RETURN NIL;
    END;

    return:=t.current.GetPosObject(x,y,type);
    IF return#NIL THEN
      RETURN return;
    END;

    object:=t.tabList;
    WHILE object#NIL DO
      return:=object.label.GetPosObject(x,y,type);
      IF return#NIL THEN
        RETURN return;
      END;
      object:=object.next;
    END;
    RETURN t.GetPosObject^(x,y,type);
  END GetPosObject;

  PROCEDURE (t : Tab) GetDnDObject*(x,y : LONGINT; drag : BOOLEAN):G.Object;

  VAR
    object : TabEntry;
    return : G.Object;

  BEGIN
    IF t.current=NIL THEN
      RETURN NIL;
    END;

    return:=t.current.GetDnDObject(x,y,drag);
    IF return#NIL THEN
      RETURN return;
    END;

    object:=t.tabList;
    WHILE object#NIL DO
      return:=object.label.GetDnDObject(x,y,drag);
      IF return#NIL THEN
        RETURN return;
      END;
      object:=object.next;
    END;
    RETURN NIL;
  END GetDnDObject;

  PROCEDURE (t : Tab) Draw*(x,y : LONGINT; draw :  D.DrawInfo);

  VAR
    count  : LONGINT;

  BEGIN
    t.Draw^(x,y,draw);

    FOR count:=1 TO t.count DO
      t.DrawTab(count,D.selected IN draw.mode);
    END;

    t.DrawTop;

    draw.PushForeground(D.shadowColor);

    (* right line *)
    draw.DrawLine(t.x+t.width-2,t.y+t.mh  ,t.x+t.width-2,t.y+t.height-3);
    draw.DrawLine(t.x+t.width-1,t.y+t.mh-1,t.x+t.width-1,t.y+t.height-2);

    (* bottom line *)
    draw.DrawLine(t.x,  t.y+t.height-1,t.x+t.width-1,t.y+t.height-1);
    draw.DrawLine(t.x+1,t.y+t.height-2,t.x+t.width-2,t.y+t.height-2);
    draw.PopForeground;

    draw.PushForeground(D.shineColor);

    (* left line *)
    draw.DrawLine(t.x  ,t.y+t.mh,t.x  ,t.y+t.height-2);
    draw.DrawLine(t.x+1,t.y+t.mh,t.x+1,t.y+t.height-3);
    draw.PopForeground;

    t.DrawObject(t.selected);
  END Draw;

  (**
    Draw the keyboard focus.
  **)

  PROCEDURE (t : Tab) DrawFocus*;

  VAR
    entry : TabEntry;

  BEGIN
    (* If our image can draw a keyboard focus, delegate it *)
    entry:=t.GetEntry(t.selected);
    IF (entry#NIL) & ~entry.label.StdFocus() THEN
      entry.label.DrawFocus;
    ELSE
      (* Delegate drawing to the baseclass *)
      t.DrawFocus^;
    END;
  END DrawFocus;

  (**
    Hide the keyboard focus.
  **)

  PROCEDURE (t : Tab) HideFocus*;

  VAR
    entry : TabEntry;

  BEGIN
    (* If our image can draw a keyboard focus, delegate it *)
    entry:=t.GetEntry(t.selected);
    IF (entry#NIL) & ~entry.label.StdFocus() THEN
      entry.label.HideFocus;
    ELSE
      (* Delegate drawing to the baseclass *)
      t.HideFocus^;
    END;
  END HideFocus;

  (**
    Must be overloaded because the baseclass implements this methods
    differently.
  **)

  PROCEDURE (t : Tab) Refresh*(x,y,w,h : LONGINT);

  BEGIN
    IF t.visible & t.Intersect(x,y,w,h) THEN
      t.Redraw;
    END;
  END Refresh;

  PROCEDURE (t : Tab) Hide*;

  VAR
    entry : TabEntry;

  BEGIN
    IF t.visible THEN
      entry:=t.tabList;
      WHILE entry#NIL DO
        entry.label.Hide;
        entry:=entry.next;
      END;
      t.current.Hide;
      t.DrawHide;  (* That is suboptimal :-( We should only clear our own frames *)
      t.Hide^;
    END;
  END Hide;

BEGIN
  NEW(prefs);
  prefs.Init;
END VOTab.