/*
    Wn: A Server for the HTTP
    File: wn/send.c
    Version 2.2.5
    
    Copyright (C) 1996-8  <by John Franks>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/


#include <string.h>
#include <memory.h>
#include <errno.h>

#ifndef NO_UNISTD_H
#include <unistd.h>
#endif

#include "wn.h"
#include "version.h"
#include "parse.h"
#include "reg.h"

#define BYTECHUNK	(256*1024)
#define MAX_REDIRECT	(10)

#ifdef USE_WN_WRITE
extern int WN_write();
#else
#define WN_write(a,b,c)               write(a,b,c)
#endif


static void	filter_open(),
		send_byterange(),
		sendsubrange();

static char	*enter_range();

extern long	atol();


void
senderr( status, msg, file)
char	*status,
	*msg,
	*file;
{
	char	buf[MIDLEN];
	int	prolog_sent;

	get_remote_info( );

	Snprintf2( outheadp->status, SMALLLEN, "%.32s %.200s", status, msg);
	mystrncpy( outheadp->list, 
		"Content-type: text/html; charset=iso-8859-1\r\n", SMALLLEN);

	if ( *status == '4') {
		/* it's a 4xx error; a client error */
		if ( streq( status, "416") ) {
			Snprintf1( outheadp->range, SMALLLEN, 
					"*/%ld", this_rp->datalen);
		}
		else {
			this_conp->keepalive = FALSE;
		}
	}

	if ( streq( status, SERV_ERR)) {
		this_conp->keepalive = FALSE;
		logerr( msg, file);
	}
	else
		writelog( this_rp, msg, file);

	prolog_sent = this_rp->status & WN_PROLOGSENT;
	clear_req( );

	this_rp->status |= prolog_sent;
	if ( !prolog_sent) {
		set_interface_root( );

		Snprintf2( buf, SMALLLEN, 
		"<html>\n<head>\n<title>%.32s %.150s</title>\n</head>\n<body>\n", 
					status, msg);
		Snprintf2( buf + strlen( buf), SMALLLEN,
			"<h2>Error code %.32s</h2>\n%.150s\n", status, msg);
		Snprintf1( buf + strlen( buf), (sizeof( buf) - strlen( buf)),
			"\n<hr>\n<address>%.32s</address>\n", VERSION);
		mystrncat( buf, "\n</body>\n</html>\n", sizeof( buf) );
		Snprintf1( this_rp->length, TINYLEN, "%d", strlen(buf));

		if (inheadp->method != HEAD)
			this_rp->status |= WN_HAS_BODY;

		http_prolog( );

		if (inheadp->method != HEAD)
			send_text_line( buf);
	}

	this_rp->status |= WN_ERROR;
	this_rp->type = RTYPE_FINISHED;

}


void
sendinfo( ip)
Request	*ip;

{
	char	*cp,
		*maintainer,
		owner[MIDLEN],
		len[TINYLEN],
		con[SMALLLEN],
		enc[SMALLLEN],
		buf[2*BIGLEN];

	int	i;

	struct tm *gmt;
	
	
	if ( *ip->length) {
		mystrncpy( len, ip->length, TINYLEN);
		*ip->length = '\0';
	}
	mystrncpy( enc, ip->encoding, SMALLLEN);
	*ip->encoding = '\0';

	mystrncpy( con, ip->content_type, SMALLLEN);
	ip->content_type = "text/html; charset=iso-8859-1";
	this_rp->status |= WN_HAS_BODY;
	http_prolog( );

	mystrncpy(buf, "<html>\n<head>\n<title>URL information </title>\n", 
			SMALLLEN );
	send_text_line( buf);
	maintainer = ( *dir_p->dir_owner ? dir_p->dir_owner : MAINTAINER);
	Snprintf1( owner, MIDLEN, "<link rev=\"made\" href=\"%.32s\">\n",
		   maintainer);
	Snprintf1(buf, MIDLEN, "%.256s</head>\n<body>\n<h2>URL information</h2>\n",
					owner);
	send_text_line( buf);

	Snprintf1(buf, MIDLEN, "<dl>\n<dt><b>Title:</b>\n<dd>%.256s\n", ip->title);
	send_text_line( buf);

	Snprintf1(buf, MIDLEN, "<dt><b> Filename:</b>\n<dd>%.128s\n", ip->basename);
	send_text_line( buf);

	if ( *ip->keywords) {
		Snprintf1(buf, MIDLEN, "<dt><b>Keywords:</b>\n<dd>%.1024s\n",
						ip->keywords);
		send_text_line( buf);
	}
	for ( i = 0; i < NUMFIELDS; i++) {
		if ( *(ip->field[i])) {
			Snprintf2(buf, MIDLEN, 
				"<dt><b>User field %d:</b>\n<dd>%.1024s\n",
							i, ip->field[i]);
			send_text_line( buf);
		}
	}

	if ( *ip->etag) {
		Snprintf1(buf, MIDLEN, "<dt><b>ETag:</b>\n<dd>%.256s\n", ip->etag);
		send_text_line( buf);
	}

	if ( ip->expires && *ip->expires) {
		Snprintf1(buf, MIDLEN, 
			"<dt><b>Expires:</b>\n<dd>%.64s\n", ip->expires);
		send_text_line( buf);
	}

	if ( *len ) {
		Snprintf1(buf, SMALLLEN, "<dt><b>Size:</b>\n<dd>%.32s\n", len);
		send_text_line( buf);
	}
	Snprintf1(buf, SMALLLEN, "<dt><b>Content-type:</b>\n<dd>%.128s\n", con);
	send_text_line( buf);

	if ( *enc) {
		Snprintf1(buf, SMALLLEN, 
				"<dt><b>Content-encoding:</b>\n<dd>%.200s\n", enc);
		send_text_line( buf);
	}

	if ( outheadp->md5 &&  (cp = strchr( outheadp->md5, ':')) != NULL ) {
		Snprintf1(buf, SMALLLEN, 
			"<dt><b>Content-MD5:</b>\n<dd>%.128s\n", ++cp);
		send_text_line( buf);
	}

	if ( ip->mod_time) {
		gmt = gmtime(&ip->mod_time);
		strftime( buf, SMALLLEN,
		"<dt><b>Last-Modified:</b>\n<dd> %a, %d %h %Y %T GMT\n", gmt);
		send_text_line( buf);
	}

	Snprintf1(buf, MIDLEN, "<dt><b>Maintainer:</b>\n<dd>%.256s\n", maintainer);
	send_text_line( buf);

	Snprintf1(buf, SMALLLEN, "</dl>\n<hr>\n<address>%.32s</address>\n",
		VERSION);
	send_text_line( buf);

	send_text_line( "\n</body>\n</html>\n");

	writelog( ip, log_m[14], ip->relpath);
}

/*
 * void file_open( ip)
 * Call check_perm() to check permissions then open a file to be served, 
 * store the FILE pointer in ip->fp.  If the string dir_p->filemod is
 * non-empty then use it as a data base module to to produce the data.
 * The data base module gets its key (which is ip->basename) from the
 * environment variable WN_KEY.
 *
 */

void
file_open( ip)
Request *ip;
{
	char	envkey[2*SMALLLEN];


	if ( !*dir_p->filemod) {
		check_perm( ip, ip->filepath);
		if ( (ip->fp = fopen( ip->filepath, "r")) == (FILE *) NULL ) {
			senderr( DENYSTATUS, err_m[1], ip->filepath);
			wn_exit( 2); /* senderr: DENYSTATUS */
		}
		ip->fptype = FRP_FILE;
		return;
	}	
	else {
		mystrncpy( envkey, "WN_KEY=", SMALLLEN);
		mystrncpy( envkey + strlen("WN_KEY="), ip->basename, SMALLLEN);
		putenv( envkey);

		if ((ip->fp = popen( dir_p->filemod, "r"))
					== (FILE *) NULL ) {
			senderr( SERV_ERR, err_m[39], dir_p->filemod);
			wn_exit( 2); /* senderr: SERV_ERR */
		}
		ip->fptype = FRP_PIPE;
	}
}

/*
 * void filter_open( ip)
 * Like file_open above, but additionally pipes the output of the 
 * file or data base module to the filter in ip->filter.  The FILE
 * pointer for the output from the filter is put in ip->fp.
 *
 */

static void
filter_open( ip)
Request *ip;
{
	char	commandbuf[2*MIDLEN],
		buf[MIDLEN],
		envkey[2*SMALLLEN];

	exec_ok( ip);
	check_perm( ip, ip->filter);
	getfpath( buf, ip->filter, ip);
	if ( !*dir_p->filemod) {
		check_perm( ip, ip->filepath);
		mystrncpy( commandbuf, buf, MIDLEN);
		mystrncat( commandbuf, " < ", 2*MIDLEN);
		mystrncat( commandbuf, ip->filepath, 2*MIDLEN);
	}
	else {
		mystrncpy( envkey, "WN_KEY=", 2*SMALLLEN);
		mystrncat( envkey, ip->basename, 2*SMALLLEN);
		putenv( envkey);
		Snprintf3( commandbuf, 2*MIDLEN,  "%.1024s %.1024s | %.1024s",
			dir_p->filemod, ip->basename, buf);
	}
	if ( (ip->fp = popen( commandbuf, "r")) == (FILE *) NULL ) {
		senderr( SERV_ERR, err_m[52], commandbuf);
		wn_exit( 2); /* senderr: SERV_ERR */
	}
	ip->fptype = FRP_PIPE;
}


/*
 * sendbin( ip)  Send a binary file.
 */

				
void
sendbin(  ip)
Request	*ip;

{
	if ( ip->filetype & WN_LINERANGE) {
		senderr( DENYSTATUS, err_m[54], ip->filepath);
		wn_abort( );
		return;
	}
	if ( ip->attributes & WN_FILTERED )
		filter_open( ip);
	else
		file_open( ip);

	if ( ip->filetype & (WN_BYTERANGE + WN_RFC_BYTERANGE)) {
		if ( !(ip->filetype & WN_RFC_BYTERANGE))
			ip->content_type = "application/octet-stream";
		send_byterange();
	}
	else {
		http_prolog( );
		send_out_fd( fileno( ip->fp));
	}

	writelog( ip, log_m[13], ip->relpath);

	if ( ip->fptype == FRP_PIPE)
		pclose( ip->fp);
	else
		fclose( ip->fp);
}

/*
 * sendtext( ip)  Send a text file.
 */

void
sendtext(  ip)
Request	*ip;
{

	static int	dontlog = 0;

	char	buf[OUT_BUFFSIZE];

	int	n,
		invalid;


	if ( ip->attributes & WN_FILTERED )
		filter_open( ip);
	else
		file_open( ip);

	if ( ip->filetype & (WN_BYTERANGE + WN_RFC_BYTERANGE + WN_LINERANGE)) {
		if ( ip->attributes & (WN_PARSE + WN_DYNAMIC + WN_FILTERED) ) {
			senderr( DENYSTATUS, err_m[94], ip->filepath);
			wn_abort( );
			return;
		}
		else if ( !(ip->filetype & WN_RFC_BYTERANGE))
			ip->content_type = (( ip->filetype & WN_TEXT )
				? "text/plain; charset=iso-8859-1" : 
				"application/octet-stream");
	}


	if ( (ip->attributes & WN_PARSE) && (ip->filetype & WN_ISHTML) ){ 
		dontlog++;
		/* Don't do http_prolog() until later */
		do_wrap( ip, SHOW_IT);
		dontlog--;
		if ( !dontlog)
			writelog( ip, log_m[13], ip->relpath);
		return;
	}
	else {
		if ( ip->filetype & (WN_BYTERANGE + WN_RFC_BYTERANGE)) {
			send_byterange();
		}
		else if ( ip->filetype & WN_LINERANGE) {
			long	startline,
				endline;
			char	*cp;

			cp = ip->range;
			enter_range( cp, &startline, &endline, &invalid);
			*ip->length ='\0';
			http_prolog( );
			if ( startline == -1 )
				logerr( err_m[93], "");
			for ( n = 1; n <= endline; n++) {
				if ( fgets( buf, OUT_BUFFSIZE, ip->fp) == NULL)
					break;
			/*
			   Note if a line is longer than OUT_BUFFSIZE so the
			   only a partial line is read, then the line count 
                           here will be wrong.  This is a bug.
			 */
				if ( n >= startline)
					send_text_line( buf);
			}
		}
		else if ( ip->type == RTYPE_MARKLINE ) {
			http_prolog( );
			send_markline_doc( ip, SHOW_IT);
		}
		else {
			http_prolog( );
			send_out_fd( fileno( ip->fp));
		}
	}

	if ( !dontlog)
		writelog( ip, log_m[13], ip->relpath);

	if ( ip->fptype == FRP_PIPE)
		pclose( ip->fp);
	else
		fclose( ip->fp);

}

void
send_text_line( line)
char	*line;
{
	send_out_mem( line, strlen(line));
}


void
sendredirect( ip, status, location)
Request	*ip;
char	*status,
	*location;
{
	static int	num = 0;
	char	buf[MIDLEN];

	num++;
	if ( num > MAX_REDIRECT) {
		senderr( SERV_ERR, err_m[55], err_m[64]);
		wn_exit( 2); /* senderr: SERV_ERR */
	}

	if ( strncasecmp( location, "<null>", 6) == 0) {
		send204( ip);
		return;
	}

	outheadp->ohstat |= OHSTAT_ISREDIR;
	if ( location != outheadp->location)
		mystrncpy( outheadp->location, location, MIDLEN);
	mystrncpy( outheadp->status, status, SMALLLEN);
	ip->content_type = ip->encoding = NULL;
	ip->attributes = ip->attrib2 = 0;
	*outheadp->list = '\0';


	Snprintf1( buf, SMALLLEN, 
	"<html>\n<head>\n<title>%.50s</title>\n</head>\n<body>\n", status);
	Snprintf1( buf + strlen( buf), SMALLLEN,
			"<h2>Redirection: %.32s</h2>\n", status);
	mystrncat( buf, "<p> This resource is located at ", sizeof(buf));
	Snprintf2( buf + strlen( buf), (sizeof( buf) - strlen( buf)),
        "<a href=\"%.128s\">%.128s.</a></p>\n", location, location);
	Snprintf1( buf + strlen( buf), (sizeof( buf) - strlen( buf)),
			"\n<hr>\n<address>%.32s</address>\n", VERSION);
	mystrncat( buf, "\n</body>\n</html>\n", sizeof( buf) );
	Snprintf1( this_rp->length, TINYLEN, "%d", strlen(buf));

	if (inheadp->method != HEAD)
			this_rp->status |= WN_HAS_BODY;

	http_prolog( );

	if (inheadp->method != HEAD)
		send_text_line( buf);

	writelog( ip, log_m[9], location);


	if ( (ip->attributes & (WN_PARSE + WN_DYNAMIC +WN_FILTERED))
				&& ( inheadp->protocol == HTTP1_0) ) {
		this_conp->keepalive = FALSE;
	}
	ip->type = RTYPE_FINISHED;
	return;
}

void
send204( ip)
Request	*ip;
{
	mystrncpy( outheadp->status, "204 No Response", SMALLLEN);

	ip->status &= ~(WN_HAS_BODY);
	http_prolog();

	writelog( ip, log_m[16], "");
	if ( (ip->attributes & (WN_PARSE + WN_DYNAMIC +WN_FILTERED))
				&& ( inheadp->protocol == HTTP1_0) ) {
		this_conp->keepalive = FALSE;
	}
	ip->type = RTYPE_FINISHED;
	return;
}


static void
send_byterange( )
{

	char	*nextrange,
		*save_content,
		sep[SMALLLEN],
		buf[SMALLLEN];

	long	startbyte,
		endbyte,
		temp_len,
		file_len,
		len_sent;

	int	invalid,
		multi = FALSE,
		first_one = TRUE;

	len_sent = 0;
	file_len = this_rp->datalen;
	nextrange = this_rp->range;

	if ( strchr( nextrange, ',') != NULL)
		multi = TRUE;

	while ( nextrange ) {
		nextrange = enter_range( nextrange, &startbyte, &endbyte, &invalid);
		if ( invalid) {
			if ( invalid == 416 )
				senderr( "416", err_m[113], "");
			if ( invalid == 400 )
				senderr( CLIENT_ERR, err_m[121], "");
			wn_exit( 2); /* senderr: CLIEN_ERR or 113 */
		}

		if ( startbyte == -1 ) {
			temp_len = endbyte;
			endbyte = file_len - 1;
			startbyte = file_len - temp_len;
		}
		else {
			if ( (endbyte == -1 ) || (endbyte >= this_rp->datalen))
				endbyte = this_rp->datalen - 1;
		}

		mystrncpy( outheadp->status,
					"206 Partial Content", SMALLLEN);

		if ( multi && (this_rp->filetype & WN_RFC_BYTERANGE) ) {
			if ( first_one) {
				first_one = FALSE;
				*(this_rp->length) = '\0';
				srand( this_conp->pid);
				Snprintf3( sep, (SMALLLEN - 64), "WN%xX%xX%xWN",
						rand(), rand(), rand());
				Snprintf1( buf, SMALLLEN,
				"multipart/x-byteranges; boundary=%.64s",
						sep);
				save_content = this_rp->content_type;
				this_rp->content_type = buf;
				http_prolog( );
				if ( this_conp->chunk_status
				     & (WN_START_CHUNK + WN_IN_CHUNK)) {
					senderr( SERV_ERR, err_m[120], "");
					wn_exit( 2); /* senderr: SERV_ERR */
				}
				this_rp->content_type = save_content;
			}
			Snprintf1( buf, SMALLLEN, "\r\n--%.64s\r\n", sep);


			send_text_line( buf);
			len_sent += strlen( buf);

			Snprintf1( buf, SMALLLEN, "Content-type: %.200s\r\n",
						this_rp->content_type);
			send_text_line( buf);
			len_sent += strlen( buf);

			Snprintf3( buf, SMALLLEN,
				"Content-Range: bytes %ld-%ld/%ld\r\n\r\n",
					startbyte, endbyte, file_len);
			send_text_line( buf);
			len_sent += strlen( buf);

			sendsubrange( startbyte, endbyte);
			len_sent += (endbyte - startbyte + 1);

			if ( nextrange == (char *)NULL) {
				Snprintf1( buf, SMALLLEN,
					 "\r\n--%.64s--\r\n", sep);
				send_text_line( buf);
				len_sent += strlen( buf);
				Snprintf1(this_rp->length, TINYLEN, "%ld", len_sent);
				return;
			}
			else
				continue;
		}
		else {
			Snprintf1( this_rp->length, TINYLEN, "%ld",
						endbyte - startbyte + 1);
			if ( this_rp->filetype & WN_RFC_BYTERANGE ) {
				Snprintf3( outheadp->range, SMALLLEN, "%ld-%ld/%ld",
						startbyte, endbyte, file_len);
			}
			http_prolog( );
			flush_outbuf( );
			sendsubrange( startbyte, endbyte);
			return;
		}
	}
}

static void
sendsubrange( start, end)
long	start,
	end;
{
	int	remlen,
		fdfp,
		len = 0;

	long	remaining;

	remaining = end - start + 1;
	fdfp = fileno( this_rp->fp);
	if ( lseek( fdfp, (off_t) start, 0 /* SEEK_SET */) < 0) {
		senderr( SERV_ERR, err_m[133], "");
		wn_exit( 2); /* senderr: SERV_ERR */
	}

	remlen = this_conp->outbuf + OUT_BUFFSIZE - this_conp->out_ptr;
	remlen = ( remlen > remaining ? (int) remaining : remlen);

	while ( TRUE) {
		len = read( fdfp, this_conp->out_ptr, remlen);
		if ( (len == -1) && (errno == EINTR))
			continue;
		if ( len <= 0 ) {
			flush_outbuf();
			break;
		}
		this_rp->logcount += (long) len;

		if ( this_conp->outbuf + OUT_BUFFSIZE <= 
					this_conp->out_ptr + len ) {  
			/* buffer is full */
			this_conp->out_ptr += len;
			flush_outbuf();
			remaining -= len;
			remlen = ( OUT_BUFFSIZE > remaining ?
					(int) remaining : OUT_BUFFSIZE);
			continue;
		}
		else {  /* buffer not full yet */
			remlen -= len;
			this_conp->out_ptr += len;
			remaining -= len;
		}
	}
	if ( (remaining > 0) || (len < 0) )
		logerr( err_m[76], "sendsubrange");


}


static char 
*enter_range(  value, startp, endp, errp )
char	*value;
long	*startp,
	*endp;
int	*errp;
{
	register char	*cp,
			*cp2;

	char		*next;

	/* if ip->param_value is "123-234" it is a file range from
	 * byte  123 to 234.  Put 123 in *startp and 234
	 * in *endp. For 123- use -1 for endp range
	 */

	cp = value;

	if ( (cp2 = strchr( cp, ',')) != NULL) {
		*cp2++ = '\0';
		next = cp2;
	}
	else
		next = (char *)NULL;

	if ( (cp2 = strchr( cp, '-')) == NULL) {
		logerr( err_m[93], cp);
		*errp = 400;
		return (next);
	}


	*cp2++ = '\0';

	if ( strchr( cp, '-')  || strchr( cp2, '-') ) {
		logerr( err_m[93], cp);
		*errp = 400;
		return (next);
	}

	*startp = ( *cp ? atol( cp ) : (-1));
	*endp = ( *cp2 ? atol( cp2 ) : this_rp->datalen);
	if ( *endp > this_rp->datalen)
		*endp = this_rp->datalen;

	if ( *startp > this_rp->datalen)
		*errp = 416;
	else if ( *startp > *endp)
		*errp = 400;
	else
		*errp = 0;
	return (next);
}




#define CHUNKPAD	(6)	/* number of bytes for chunk size */
				/* CHUNKPAD = 4 for digits + 2 for CRLF */
void
send_out_fd( fd )
int	fd;
{

	int		remlen,
			len;

	remlen = this_conp->outbuf + OUT_BUFFSIZE - this_conp->out_ptr;

	while ( TRUE) {
		if ( this_conp->chunk_status & WN_START_CHUNK) {
			/* set up chunking */
			this_conp->chunksize = 0;
			if ( remlen < OUT_BUFFSIZE/4 ) {
				flush_outbuf();
				remlen = OUT_BUFFSIZE;
			}

			this_conp->chunksize_ptr = this_conp->out_ptr;
			this_conp->out_ptr += CHUNKPAD;
			len = read( fd, this_conp->out_ptr, remlen);

			if ( len > 0 ) {
				this_conp->chunk_status &= ~(WN_START_CHUNK);
				this_conp->chunk_status |= WN_IN_CHUNK;
				this_rp->logcount += CHUNKPAD;
			}
			else {  /* len <= 0 */
				this_conp->out_ptr -= CHUNKPAD;
				/* undo this */
				if ( (len == -1) && (errno == EINTR))
					continue;
				else
					break;
			}
		}
		else {
			len = read( fd, this_conp->out_ptr, remlen);
			if ( (len == -1) && (errno == EINTR))
				continue;
			if ( len <= 0 )
				break;
		}

		this_rp->logcount += (long) len;
		this_conp->out_ptr += len;
		if ( this_conp->chunk_status & WN_IN_CHUNK) {
			this_conp->chunksize += (long)len;
		}
		if ( len == remlen ) {  /* buffer is full */
			flush_outbuf();
			remlen = OUT_BUFFSIZE;
			continue;
		}
		else {  /* buffer not full yet */
			remlen -= len;
			if ( this_rp->attributes & WN_UNBUFFERED)
				flush_outbuf();
		}
	}
	if ( len < 0 )
		logerr( err_m[76], "send_out_fd");
}

void
send_out_mem( buf, len)
char	*buf;
int	len;
{
	register char	*cp,
			*cp2,
			*end;

	int		remlen;
	end = this_conp->outbuf + OUT_BUFFSIZE;
	cp = buf;

	this_rp->logcount += (long) len;
	while ( len > 0 ){
		if ( this_conp->chunk_status & WN_START_CHUNK) {
			/* set up chunking */
			if ( (end - this_conp->out_ptr) < OUT_BUFFSIZE/4 ) {
				flush_outbuf();
			}
			this_conp->chunksize = 0;
			this_conp->chunk_status &= ~(WN_START_CHUNK);
			this_conp->chunk_status |= WN_IN_CHUNK;
			this_conp->chunksize_ptr = this_conp->out_ptr;
			this_conp->out_ptr += CHUNKPAD;
			this_rp->logcount += CHUNKPAD;
		}
		cp2 = this_conp->out_ptr;
		if ( end > cp2 + len ) {
			/* it all fits in buffer */
			memcpy( cp2, cp, len);
			this_conp->out_ptr += len;
			if ( this_conp->chunk_status & WN_IN_CHUNK)
				this_conp->chunksize += (long)len;
			if ( this_rp->attributes & WN_UNBUFFERED)
				flush_outbuf();
			return;
		}
		else {
			remlen = this_conp->outbuf
					+ OUT_BUFFSIZE - cp2;
			len -= remlen;
			memcpy( cp2, cp, remlen);

			if ( this_conp->chunk_status & WN_IN_CHUNK)
				this_conp->chunksize += (long)remlen;

			cp += remlen;
			this_conp->out_ptr = end;
			flush_outbuf();
		}
	}
}


void
flush_outbuf( )
{

	int		fdstdout,
			len,
			n;

	register char	*cp;

	if ( this_conp->out_ptr == this_conp->outbuf)
		return;
	if ( this_conp->chunk_status & WN_IN_CHUNK) {
		cp = this_conp->chunksize_ptr;

		Snprintf1( cp, CHUNKPAD, "%.4lX\r", this_conp->chunksize);
		/* 4 in %.4lX is CHUNKPAD - 2 */

		cp += strlen(cp);
		*cp = '\n';
		this_conp->chunksize = 0;
		*this_conp->out_ptr++ = '\r';
		*this_conp->out_ptr++ = '\n';
		*this_conp->out_ptr = '\0';
		this_conp->chunk_status &= ~(WN_IN_CHUNK);
		this_conp->chunk_status |= WN_START_CHUNK;
	}


	cp = this_conp->outbuf;
	len = this_conp->out_ptr - cp;

	fdstdout = fileno( stdout);

	while ( (n = WN_write( fdstdout, cp, len)) < len) {

		if ( n == -1 && errno == EINTR) 
			continue;
		if ( n <= 0) {
			if ( n == -1 && errno != EPIPE) {
				char   buf[TINYLEN];
				Snprintf1( buf, TINYLEN, "flush, errno = %d", errno);
				logerr( err_m[75], buf);
			}
			break;
		}
		len -= n;
		cp += n;
		this_conp->bytecount += n;
	}
	if ( n >= 0 ) {
		this_conp->bytecount += n;
	}

	if ( this_conp->bytecount >= BYTECHUNK) {
		alarm( TRANSACTION_TIMEOUT);
		this_conp->bytecount = 0L;
	}
	this_conp->out_ptr = this_conp->outbuf;
}

/* Chunking is automatically done by buffer routines (flush_outbuf,
 * send_out_fd() and send_out_mem() ) whenever chunk_status & WN_USE_CHUNK
 * is true.  There are two disjoint states for chunking WN_START_CHUNK and
 * WN_IN_CHUNK.  With the first we are ready to start a new chunk (any
 * previous one has been completed.  In this state when one of the send_out_*
 * functions is called space is reserved in the buffer to place the
 * chunk length,  WN_IN_CHUNK is set, an a chunksize count is started.
 * When in the state WN_IN_CHUNK, any calls to send_out_* put bytes
 * in the buffer and update the chunksize.  When the buffer is full
 * flush_outbuf() is called.  It completes a chunk by filling in the
 * the chunk size (at chunksize_ptr), terminating the chunk with
 * CRLF, and setting the state to WN_START_CHUNK, before the buffer is
 * flushed.  Finally when the document being sent is complete, a
 * call to end_chunking() completes the current chunk if necessary,
 * i.e. if WN_IN_CHUNK, appends the zero size chunk indicating the
 * end of chunking and reinitializes chunk_status, turning off all
 * bits.
 * Currently chunk size is formatted as %.4lX, e.g. 02FF.  The maximum
 * chunk size is OUT_BUFFSIZE which must be less than FFFF hex.
 */

void
end_chunking()
{
	register char	*cp,
			*cp2;

	if ( !this_conp->chunk_status & WN_USE_CHUNK) {
		logerr( err_m[116], "");
		return;
	}
	cp = this_conp->out_ptr;
	if ( this_conp->chunk_status & WN_IN_CHUNK) {
		/* finish the chunk we're doing */
		cp2 = this_conp->chunksize_ptr;

		Snprintf1( cp2, CHUNKPAD, "%.4lX\r", this_conp->chunksize);
		/* 4 in %.4lX is CHUNKPAD - 2 */

		cp2 += strlen(cp2);
		*cp2 = '\n';

		*cp++ = '\r';
		*cp++ = '\n';
	}
	this_conp->chunksize = 0;
	this_conp->chunk_status = 0;
	/* add last (zero size) chunk */
	*cp++ = '0';
	*cp++ = '\r';
	*cp++ = '\n';
	/* "trailer" (if any) should go here */
	*cp++ = '\r';
	*cp++ = '\n';
	*cp = '\0';
	this_conp->out_ptr = cp;
}
