/* vi: set sw=4 ts=4:
 *
 * Copyright (C) 2001 - 2009 Christian Hohnstaedt.
 *
 * All rights reserved.
 */

#include "x509name.h"
#include "base.h"
#include "func.h"
#include <openssl/asn1.h>
#include <openssl/err.h>
#include "exception.h"

x509name::x509name()
{
	xn = X509_NAME_new();
}

x509name::x509name(const X509_NAME *n)
{
	xn = X509_NAME_dup((X509_NAME *)n);
}

x509name::x509name(const x509name &n)
{
	xn = NULL;
	set(n.xn);
}

x509name::~x509name()
{
	X509_NAME_free(xn);
}

x509name &x509name::set(const X509_NAME *n)
{
	if (xn != NULL)
		X509_NAME_free(xn);
	xn = X509_NAME_dup((X509_NAME *)n);
	return *this;
}


QString x509name::oneLine(unsigned long flags) const
{
	QString ret;
	long l;
	const char *p;
	BIO *mem = BIO_new(BIO_s_mem());
	X509_NAME_print_ex(mem, xn, 0, flags);
	l = BIO_get_mem_data(mem, &p);
	ret = ret.fromUtf8(p,l);
	BIO_free(mem);
	return ret;
}

QString x509name::getEntryByNid(int nid) const
{
	int i = X509_NAME_get_index_by_NID(xn, nid, -1);
	if (i < 0)
		return QString::null;
	return getEntry(i);
}

QString x509name::getMostPopular() const
{
	static const int nids[] = { NID_commonName, NID_pkcs9_emailAddress,
			NID_organizationalUnitName, NID_organizationName };
	int pos = -1;

	for (unsigned i = 0; i < ARRAY_SIZE(nids) && pos < 0; i++) {
		pos = X509_NAME_get_index_by_NID(xn, nids[i], -1);
	}
	if (pos < 0)
		pos = 0;
	return getEntry(pos);
}

QString x509name::getEntry(int i) const
{
	QString ret;
	ASN1_STRING *d;

	if ( i<0 || i>entryCount() )
		return ret;

	d = X509_NAME_ENTRY_get_data(X509_NAME_get_entry(xn,i));

	return asn1ToQString(d);
}

QString x509name::getEntryTag(int i) const
{
	QString s = QObject::tr("Invalid");
	ASN1_STRING *d;

	if (i<0 || i>=entryCount())
		i = entryCount() - 1;
	d = X509_NAME_ENTRY_get_data(X509_NAME_get_entry(xn,i));

	if (!d)
		return s;

	s = ASN1_tag2str(d->type);
	return s;
}

QString x509name::popEntryByNid(int nid)
{
	int i = X509_NAME_get_index_by_NID(xn, nid, -1);
	if (i < 0)
		return QString::null;
	QString n = getEntry(i);
	X509_NAME_delete_entry(xn, i);
	return n;
}

QStringList x509name::entryList(int i) const
{
	QStringList sl;
	int n = nid(i);
	if (n == NID_undef) {
		QString oid = getOid(i);
		sl << oid << oid;
	} else {
		sl << OBJ_nid2sn(n) << OBJ_nid2ln(n);
	}
	sl << getEntry(i) << getEntryTag(i);
	return sl;
}

int x509name::nid(int i) const
{
	X509_NAME_ENTRY *ne;

	ne = sk_X509_NAME_ENTRY_value(xn->entries, i);
	if (ne == NULL)
		return NID_undef;
	return OBJ_obj2nid(ne->object);
}

QString x509name::getOid(int i) const
{
	X509_NAME_ENTRY *ne;
	char buf[256];
	int len;

	ne = sk_X509_NAME_ENTRY_value(xn->entries, i);
	if (ne == NULL)
		return QString();
	len = OBJ_obj2txt(buf, 256, ne->object, 1);
	return QString::fromAscii(buf, len);
}

const unsigned char *x509name::d2i(const unsigned char *p, int size)
{
	X509_NAME *xn_sik = xn;
	xn = D2I_CLASH(d2i_X509_NAME, NULL, &p, size);
	if (xn == NULL)
		xn = xn_sik;
	else
		X509_NAME_free(xn_sik);
	return p;
}

unsigned char *x509name::i2d(unsigned char *p)
{
	i2d_X509_NAME(xn, &p);
	return p;
}

bool x509name::operator == (const x509name &x) const
{
	return (X509_NAME_cmp(xn, x.xn) == 0);
}

x509name &x509name::operator = (const x509name &x)
{
	set(x.xn);
	return *this;
}

int x509name::entryCount() const
{
	return  X509_NAME_entry_count(xn);
}

int x509name::getNidByName(const QString &nid_name)
{
	return OBJ_txt2nid(nid_name.toAscii());
}

void x509name::addEntryByNid(int nid, const QString entry)
{
	if (entry.isEmpty())
		return;
	ASN1_STRING *a = QStringToAsn1(entry, nid);
	if (!a) {
		QString error = QString(OBJ_nid2ln(nid)) + ":\n";
		while (int i = ERR_get_error() ) {
			fprintf(stderr, "OpenSSL error: %s\n", ERR_error_string(i ,NULL) );
			error += ERR_error_string(i, NULL);
			error += "\n";
		}
		throw errorEx(error, "x509name");
		return;
	}
	X509_NAME_add_entry_by_NID(xn, nid, a->type, a->data, a->length, -1, 0);
	ASN1_STRING_free(a);
}

X509_NAME *x509name::get() const
{
	return X509_NAME_dup(xn);
}

int x509name::derSize() const
{
	return i2d_X509_NAME(xn, NULL);
}
