// This is adapted from 'adie', a demo text editor found
// in the FOX library and written by Jeroen van der Zijp.

#include "config.h"
#include "i18n.h"

#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>

#include <fox-1.6/fx.h>
#include <fox-1.6/fxkeys.h>
#include <fox-1.6/FXPNGIcon.h>
#include <fox-1.6/FX88591Codec.h>
#include <fox-1.6/FXUTF16Codec.h>

#include "xfedefs.h"
#include "icons.h"
#include "FileDialog.h"
#include "FontDialog.h"
#include "MessageBox.h"
#include "InputDialog.h"
#include "EditWindow.h"
#include "XFileWrite.h"



extern FXbool allowPopupScroll;


FXIMPLEMENT_ABSTRACT(FXTextCommand,FXCommand,NULL,0)

// Return size of record plus any data kept here
FXuint FXTextCommand::size() const
{
    return sizeof(FXTextCommand)+ndel;
}


FXIMPLEMENT_ABSTRACT(FXTextInsert,FXTextCommand,NULL,0)

// Insert command
FXTextInsert::FXTextInsert(FXText* txt,FXint p,FXint ni,const FXchar* ins):FXTextCommand(txt,p,0,ni)
{
    FXMALLOC(&buffer,FXchar,ni);
    memcpy(buffer,ins,ni);
}


// Undo an insert removes the inserted text
void FXTextInsert::undo()
{
    text->removeText(pos,nins,TRUE);
    text->setCursorPos(pos);
    text->makePositionVisible(pos);
}


// Redo an insert inserts the same old text again
void FXTextInsert::redo()
{
    text->insertText(pos,buffer,nins,TRUE);
    text->setCursorPos(pos+nins);
    text->makePositionVisible(pos+nins);
}


FXIMPLEMENT_ABSTRACT(FXTextDelete,FXTextCommand,NULL,0)

// Delete command
FXTextDelete::FXTextDelete(FXText* txt,FXint p,FXint nd,const FXchar* del):FXTextCommand(txt,p,nd,0)
{
    FXMALLOC(&buffer,FXchar,nd);
    memcpy(buffer,del,nd);
}


// Undo a delete reinserts the old text
void FXTextDelete::undo()
{
    text->insertText(pos,buffer,ndel,TRUE);
    text->setCursorPos(pos+ndel);
    text->makePositionVisible(pos+ndel);
}


// Redo a delete removes it again
void FXTextDelete::redo()
{
    text->removeText(pos,ndel,TRUE);
    text->setCursorPos(pos);
    text->makePositionVisible(pos);
}


FXIMPLEMENT_ABSTRACT(FXTextReplace,FXTextCommand,NULL,0)

// Replace command
FXTextReplace::FXTextReplace(FXText* txt,FXint p,FXint nd,FXint ni,const FXchar* del,const FXchar* ins):FXTextCommand(txt,p,nd,ni)
{
    FXMALLOC(&buffer,FXchar,nd+ni);
    memcpy(buffer,del,nd);
    memcpy(buffer+nd,ins,ni);
}


// Undo a replace reinserts the old text
void FXTextReplace::undo()
{
    text->replaceText(pos,nins,buffer,ndel,TRUE);
    text->setCursorPos(pos+ndel);
    text->makePositionVisible(pos+ndel);
}


// Redo a replace reinserts the new text
void FXTextReplace::redo()
{
    text->replaceText(pos,ndel,buffer+ndel,nins,TRUE);
    text->setCursorPos(pos+nins);
    text->makePositionVisible(pos+nins);
}



// Preferences class

// Map
FXDEFMAP(Preferences) PreferencesMap[]=
{
	FXMAPFUNC(SEL_COMMAND,Preferences::ID_ACCEPT,Preferences::onCmdAccept),
	FXMAPFUNC(SEL_COMMAND,Preferences::ID_CANCEL,Preferences::onCmdCancel),
	FXMAPFUNC(SEL_COMMAND,Preferences::ID_WHEELADJUST,Preferences::onCmdWheelAdjust),
	FXMAPFUNC(SEL_UPDATE,Preferences::ID_WHEELADJUST,Preferences::onUpdWheelAdjust),
	FXMAPFUNC(SEL_COMMAND,Preferences::ID_TEXT_BACK,Preferences::onCmdTextBackColor),
	FXMAPFUNC(SEL_CHANGED,Preferences::ID_TEXT_BACK,Preferences::onCmdTextBackColor),
	FXMAPFUNC(SEL_UPDATE,Preferences::ID_TEXT_BACK,Preferences::onUpdTextBackColor),
	FXMAPFUNC(SEL_COMMAND,Preferences::ID_TEXT_FORE,Preferences::onCmdTextForeColor),
	FXMAPFUNC(SEL_CHANGED,Preferences::ID_TEXT_FORE,Preferences::onCmdTextForeColor),
	FXMAPFUNC(SEL_UPDATE,Preferences::ID_TEXT_FORE,Preferences::onUpdTextForeColor),
	FXMAPFUNC(SEL_COMMAND,Preferences::ID_TEXT_SELBACK,Preferences::onCmdTextSelBackColor),
	FXMAPFUNC(SEL_CHANGED,Preferences::ID_TEXT_SELBACK,Preferences::onCmdTextSelBackColor),
	FXMAPFUNC(SEL_UPDATE,Preferences::ID_TEXT_SELBACK,Preferences::onUpdTextSelBackColor),
	FXMAPFUNC(SEL_COMMAND,Preferences::ID_TEXT_SELFORE,Preferences::onCmdTextSelForeColor),
	FXMAPFUNC(SEL_CHANGED,Preferences::ID_TEXT_SELFORE,Preferences::onCmdTextSelForeColor),
	FXMAPFUNC(SEL_UPDATE,Preferences::ID_TEXT_SELFORE,Preferences::onUpdTextSelForeColor),
	FXMAPFUNC(SEL_COMMAND,Preferences::ID_TEXT_HILITEBACK,Preferences::onCmdTextHiliteBackColor),
	FXMAPFUNC(SEL_CHANGED,Preferences::ID_TEXT_HILITEBACK,Preferences::onCmdTextHiliteBackColor),
	FXMAPFUNC(SEL_UPDATE,Preferences::ID_TEXT_HILITEBACK,Preferences::onUpdTextHiliteBackColor),
	FXMAPFUNC(SEL_COMMAND,Preferences::ID_TEXT_HILITEFORE,Preferences::onCmdTextHiliteForeColor),
	FXMAPFUNC(SEL_CHANGED,Preferences::ID_TEXT_HILITEFORE,Preferences::onCmdTextHiliteForeColor),
	FXMAPFUNC(SEL_UPDATE,Preferences::ID_TEXT_HILITEFORE,Preferences::onUpdTextHiliteForeColor),
	FXMAPFUNC(SEL_COMMAND,Preferences::ID_TEXT_CURSOR,Preferences::onCmdTextCursorColor),
	FXMAPFUNC(SEL_CHANGED,Preferences::ID_TEXT_CURSOR,Preferences::onCmdTextCursorColor),
	FXMAPFUNC(SEL_UPDATE,Preferences::ID_TEXT_CURSOR,Preferences::onUpdTextCursorColor),
	FXMAPFUNC(SEL_COMMAND,Preferences::ID_TEXT_NUMBACK,Preferences::onCmdTextBarColor),
	FXMAPFUNC(SEL_CHANGED,Preferences::ID_TEXT_NUMBACK,Preferences::onCmdTextBarColor),
	FXMAPFUNC(SEL_UPDATE,Preferences::ID_TEXT_NUMBACK,Preferences::onUpdTextBarColor),
	FXMAPFUNC(SEL_COMMAND,Preferences::ID_TEXT_NUMFORE,Preferences::onCmdTextNumberColor),
	FXMAPFUNC(SEL_CHANGED,Preferences::ID_TEXT_NUMFORE,Preferences::onCmdTextNumberColor),
	FXMAPFUNC(SEL_UPDATE,Preferences::ID_TEXT_NUMFORE,Preferences::onUpdTextNumberColor),
};

// Object implementation
FXIMPLEMENT(Preferences,DialogBox,PreferencesMap,ARRAYNUMBER(PreferencesMap))

// Construct
Preferences::Preferences(EditWindow *owner):DialogBox(owner,_("XFileWrite Preferences"),DECOR_TITLE|DECOR_BORDER)
{
	// Get the editor text widget from the owner
	editwin=owner;
	editor=owner->getEditor();
	
    // Set title
    setTitle(_("XFileWrite Preferences"));

    // Buttons
    FXHorizontalFrame *buttons=new FXHorizontalFrame(this,LAYOUT_SIDE_BOTTOM|LAYOUT_FILL_X,0,0,0,0,10,10,5,5);

    // Contents
    FXHorizontalFrame *contents=new FXHorizontalFrame(this,LAYOUT_SIDE_TOP|FRAME_NONE|LAYOUT_FILL_X|LAYOUT_FILL_Y|PACK_UNIFORM_WIDTH);

    // Accept
    FXButton *ok = new FXButton(buttons,_("&Accept"),NULL,this,Preferences::ID_ACCEPT,FRAME_RAISED|FRAME_THICK|LAYOUT_RIGHT|LAYOUT_CENTER_Y,0,0,0,0,20,20);
    ok->addHotKey(KEY_Return);

    // Cancel
    new FXButton(buttons,_("&Cancel"),NULL,this,Preferences::ID_CANCEL,FRAME_RAISED|FRAME_THICK|LAYOUT_RIGHT|LAYOUT_CENTER_Y,0,0,0,0,20,20);

    // Switcher
    FXTabBook *tabbook = new FXTabBook(contents,NULL,0,LAYOUT_FILL_X|LAYOUT_FILL_Y|LAYOUT_RIGHT);

    // First tab - Editor
    new FXTabItem(tabbook,_("&Editor"),NULL);
    FXVerticalFrame *editor=new FXVerticalFrame(tabbook,FRAME_THICK|FRAME_RAISED);
    FXGroupBox *group=new FXGroupBox(editor,_("Text"),GROUPBOX_TITLE_LEFT|FRAME_GROOVE|LAYOUT_FILL_X);
    FXMatrix *matrix = new FXMatrix(group,2,MATRIX_BY_COLUMNS|LAYOUT_SIDE_TOP|LAYOUT_FILL_X|LAYOUT_FILL_Y);
    group=new FXGroupBox(editor,_("Mouse"),GROUPBOX_TITLE_LEFT|FRAME_GROOVE|LAYOUT_FILL_X|LAYOUT_FILL_Y);
    FXMatrix *matrix3 = new FXMatrix(group,2,MATRIX_BY_COLUMNS|LAYOUT_SIDE_TOP|LAYOUT_FILL_X|LAYOUT_FILL_Y);
    
	new FXLabel(matrix,_("Wrap margin:"),NULL,JUSTIFY_LEFT|LAYOUT_FILL_COLUMN|LAYOUT_FILL_ROW);
    wrapmargin = new FXTextField(matrix,10,NULL,0,TEXTFIELD_INTEGER|JUSTIFY_RIGHT|FRAME_SUNKEN|FRAME_THICK|LAYOUT_CENTER_Y|LAYOUT_FILL_X|LAYOUT_FILL_ROW,0,0,0,0, 2,2,1,1);
    FXint wrapcols=getApp()->reg().readIntEntry("SETTINGS","wrapcols",80);
	wrapmargin->setText(FXStringVal(wrapcols));
	
	new FXLabel(matrix,_("Tabulation size:"),NULL,JUSTIFY_LEFT|LAYOUT_FILL_COLUMN|LAYOUT_FILL_ROW);
    tabsize = new FXTextField(matrix,10,NULL,0,TEXTFIELD_INTEGER|JUSTIFY_RIGHT|FRAME_SUNKEN|FRAME_THICK|LAYOUT_CENTER_Y|LAYOUT_FILL_X|LAYOUT_FILL_ROW,0,0,0,0, 2,2,1,1);
    FXint tabcols=getApp()->reg().readIntEntry("SETTINGS","tabcols",80);
	tabsize->setText(FXStringVal(tabcols));

    new FXLabel(matrix,_("Strip carriage returns:               "),NULL,JUSTIFY_LEFT|LAYOUT_FILL_COLUMN|LAYOUT_FILL_ROW);
    stripcr = new FXCheckButton(matrix,FXString::null,NULL,0,LAYOUT_LEFT|LAYOUT_CENTER_Y|LAYOUT_FILL_ROW,0,0,0,0, 0,0,0,0);
    FXint stripreturn=getApp()->reg().readIntEntry("SETTINGS","stripreturn",FALSE);
	stripcr->setCheck(stripreturn);

    new FXLabel(matrix3,_("Mouse scrolling speed:"),NULL,JUSTIFY_LEFT|LAYOUT_FILL_COLUMN|LAYOUT_FILL_ROW);
    FXSpinner* spinner=new FXSpinner(matrix3,3,this,Preferences::ID_WHEELADJUST,JUSTIFY_RIGHT|FRAME_SUNKEN|FRAME_THICK|LAYOUT_FILL_X|LAYOUT_FILL_ROW,0,0,0,0, 2,2,1,1);
    spinner->setRange(1,100);

    // Second tab - Colors
    new FXTabItem(tabbook,_("&Colors"),NULL);
    FXVerticalFrame *colors=new FXVerticalFrame(tabbook,FRAME_THICK|FRAME_RAISED);
    group=new FXGroupBox(colors,_("Text"),GROUPBOX_TITLE_LEFT|FRAME_GROOVE|LAYOUT_FILL_X);
    FXMatrix *matrix1 = new FXMatrix(group,2,MATRIX_BY_COLUMNS|LAYOUT_SIDE_TOP|LAYOUT_FILL_X|LAYOUT_FILL_Y);
    group=new FXGroupBox(colors,_("Lines"),GROUPBOX_TITLE_LEFT|FRAME_GROOVE|LAYOUT_FILL_X);
    FXMatrix *matrix2 = new FXMatrix(group,2,MATRIX_BY_COLUMNS|LAYOUT_SIDE_TOP|LAYOUT_FILL_X|LAYOUT_FILL_Y);

    new FXLabel(matrix1,_("Background:"),NULL,JUSTIFY_LEFT|LAYOUT_FILL_COLUMN|LAYOUT_FILL_ROW);
    new FXColorWell(matrix1,FXRGB(0,0,0),this,Preferences::ID_TEXT_BACK,FRAME_SUNKEN|FRAME_THICK|LAYOUT_LEFT|LAYOUT_CENTER_Y|LAYOUT_FIX_WIDTH|LAYOUT_FIX_HEIGHT|LAYOUT_FILL_ROW,0,0,40,24);

    new FXLabel(matrix1,_("Text:"),NULL,JUSTIFY_LEFT|LAYOUT_FILL_COLUMN|LAYOUT_FILL_ROW);
    new FXColorWell(matrix1,FXRGB(0,0,0),this,Preferences::ID_TEXT_FORE,FRAME_SUNKEN|FRAME_THICK|LAYOUT_LEFT|LAYOUT_CENTER_Y|LAYOUT_FIX_WIDTH|LAYOUT_FIX_HEIGHT|LAYOUT_FILL_ROW,0,0,40,24);

    new FXLabel(matrix1,_("Selected text background:"),NULL,JUSTIFY_LEFT|LAYOUT_FILL_COLUMN|LAYOUT_FILL_ROW);
    new FXColorWell(matrix1,FXRGB(0,0,0),this,Preferences::ID_TEXT_SELBACK,FRAME_SUNKEN|FRAME_THICK|LAYOUT_LEFT|LAYOUT_CENTER_Y|LAYOUT_FIX_WIDTH|LAYOUT_FIX_HEIGHT|LAYOUT_FILL_ROW,0,0,40,24);

    new FXLabel(matrix1,_("Selected text:"),NULL,JUSTIFY_LEFT|LAYOUT_FILL_COLUMN|LAYOUT_FILL_ROW);
    new FXColorWell(matrix1,FXRGB(0,0,0),this,Preferences::ID_TEXT_SELFORE,FRAME_SUNKEN|FRAME_THICK|LAYOUT_LEFT|LAYOUT_CENTER_Y|LAYOUT_FIX_WIDTH|LAYOUT_FIX_HEIGHT|LAYOUT_FILL_ROW,0,0,40,24);

    new FXLabel(matrix1,_("Highlighted text background:"),NULL,JUSTIFY_LEFT|LAYOUT_FILL_COLUMN|LAYOUT_FILL_ROW);
    new FXColorWell(matrix1,FXRGB(0,0,0),this,Preferences::ID_TEXT_HILITEBACK,FRAME_SUNKEN|FRAME_THICK|LAYOUT_LEFT|LAYOUT_CENTER_Y|LAYOUT_FIX_WIDTH|LAYOUT_FIX_HEIGHT|LAYOUT_FILL_ROW,0,0,40,24);

    new FXLabel(matrix1,_("Highlighted text:"),NULL,JUSTIFY_LEFT|LAYOUT_FILL_COLUMN|LAYOUT_FILL_ROW);
    new FXColorWell(matrix1,FXRGB(0,0,0),this,Preferences::ID_TEXT_HILITEFORE,FRAME_SUNKEN|FRAME_THICK|LAYOUT_LEFT|LAYOUT_CENTER_Y|LAYOUT_FIX_WIDTH|LAYOUT_FIX_HEIGHT|LAYOUT_FILL_ROW,0,0,40,24);

    new FXLabel(matrix1,_("Cursor:"),NULL,JUSTIFY_LEFT|LAYOUT_FILL_COLUMN|LAYOUT_FILL_ROW);
    new FXColorWell(matrix1,FXRGB(0,0,0),this,Preferences::ID_TEXT_CURSOR,FRAME_SUNKEN|FRAME_THICK|LAYOUT_LEFT|LAYOUT_CENTER_Y|LAYOUT_FIX_WIDTH|LAYOUT_FIX_HEIGHT|LAYOUT_FILL_ROW,0,0,40,24);

    new FXLabel(matrix2,_("Line numbers background:"),NULL,JUSTIFY_LEFT|LAYOUT_FILL_COLUMN|LAYOUT_FILL_ROW);
    new FXColorWell(matrix2,FXRGB(0,0,0),this,Preferences::ID_TEXT_NUMBACK,FRAME_SUNKEN|FRAME_THICK|LAYOUT_LEFT|LAYOUT_CENTER_Y|LAYOUT_FIX_WIDTH|LAYOUT_FIX_HEIGHT|LAYOUT_FILL_ROW,0,0,40,24);

    new FXLabel(matrix2,_("Line numbers foreground:"),NULL,JUSTIFY_LEFT|LAYOUT_FILL_COLUMN|LAYOUT_FILL_ROW);
    new FXColorWell(matrix2,FXRGB(0,0,0),this,Preferences::ID_TEXT_NUMFORE,FRAME_SUNKEN|FRAME_THICK|LAYOUT_LEFT|LAYOUT_CENTER_Y|LAYOUT_FIX_WIDTH|LAYOUT_FIX_HEIGHT|LAYOUT_FILL_ROW,0,0,40,24);

}

long Preferences::onCmdAccept(FXObject* o,FXSelector s,void* p)
{
	// Update preferences to their new values
	
	editor->setWrapColumns(FXIntVal(wrapmargin->getText()));	
	getApp()->reg().writeIntEntry("SETTINGS","wrapcols",FXIntVal(wrapmargin->getText()));
	
	editor->setTabColumns(FXIntVal(tabsize->getText()));	
	getApp()->reg().writeIntEntry("SETTINGS","tabcols",FXIntVal(tabsize->getText()));

	editor->setTabColumns(FXIntVal(tabsize->getText()));	
	getApp()->reg().writeIntEntry("SETTINGS","tabcols",FXIntVal(tabsize->getText()));

    editwin->setStripcr(stripcr->getCheck());
	getApp()->reg().writeIntEntry("SETTINGS","stripreturn",stripcr->getCheck());


	// Finally, update the registry
	getApp()->reg().write();
	
    DialogBox::onCmdAccept(o,s,p);
	
	return 1;
}


long Preferences::onCmdCancel(FXObject* o,FXSelector s,void* p)
{
	// Reset preferences to their previous values
	
	// First tab - Editor
	wrapmargin->setText(wrapmargin_prev);
	tabsize->setText(tabsize_prev);
	stripcr->setCheck(stripcr_prev);
	getApp()->setWheelLines(value_prev);	

	// Second tab - Colors
	editor->setTextColor(textcolor_prev);
	editor->setBackColor(backcolor_prev);	
	editor->setSelTextColor(seltextcolor_prev);
	editor->setSelBackColor(selbackcolor_prev);
	editor->setHiliteTextColor(hilitetextcolor_prev);	
	editor->setHiliteBackColor(hilitebackcolor_prev);	
	editor->setCursorColor(cursorcolor_prev);
	editor->setBarColor(barcolor_prev);
	editor->setNumberColor(numbercolor_prev);	

    DialogBox::onCmdCancel(o,s,p);
    return 1;
}


// Execute dialog box modally
FXuint Preferences::execute(FXuint placement)
{
	// Save current preferences to restore them if cancel is pressed
	
	// First tab - Editor
	wrapmargin_prev=wrapmargin->getText();
	tabsize_prev=tabsize->getText();
	stripcr_prev=stripcr->getCheck();
	value_prev=getApp()->getWheelLines();
	
	// Second tab - Colors
	textcolor_prev=editor->getTextColor();	
	backcolor_prev=editor->getBackColor();	
	seltextcolor_prev=editor->getSelTextColor();
	selbackcolor_prev=editor->getSelBackColor();
	hilitetextcolor_prev=editor->getHiliteTextColor();	
	hilitebackcolor_prev=editor->getHiliteBackColor();	
	cursorcolor_prev=editor->getCursorColor();
	barcolor_prev=editor->getBarColor();
	numbercolor_prev=editor->getNumberColor();	
	
	// Display dialog
	create();
    show(placement);
	getApp()->refresh();
    return getApp()->runModalFor(this);
}

// Set scroll wheel lines (Mathew Robertson <mathew@optushome.com.au>)
long Preferences::onCmdWheelAdjust(FXObject* sender,FXSelector,void*)
{
    FXuint value;
    sender->handle(this,FXSEL(SEL_COMMAND,ID_GETINTVALUE),(void*)&value);
    getApp()->setWheelLines(value);
	getApp()->reg().write();
    return 1;
}


// Update the wheel lines button
long Preferences::onUpdWheelAdjust(FXObject* sender,FXSelector,void*)
{
    FXuint value=getApp()->getWheelLines();
	sender->handle(this,FXSEL(SEL_COMMAND,ID_SETINTVALUE),(void*)&value);
    return 1;
}

// Change text color
long Preferences::onCmdTextForeColor(FXObject*,FXSelector,void* ptr)
{
    editor->setTextColor((FXColor)(FXuval)ptr);
    return 1;
}

// Update text color
long Preferences::onUpdTextForeColor(FXObject* sender,FXSelector,void*)
{
    FXColor color=editor->getTextColor();
    sender->handle(this,FXSEL(SEL_COMMAND,ID_SETINTVALUE),(void*)&color);
    return 1;
}


// Change text background color
long Preferences::onCmdTextBackColor(FXObject*,FXSelector,void* ptr)
{
    editor->setBackColor((FXColor)(FXuval)ptr);
    return 1;
}

// Update background color
long Preferences::onUpdTextBackColor(FXObject* sender,FXSelector,void*)
{
    FXColor color=editor->getBackColor();
    sender->handle(this,FXSEL(SEL_COMMAND,ID_SETINTVALUE),(void*)&color);
    return 1;
}


// Change selected text foreground color
long Preferences::onCmdTextSelForeColor(FXObject*,FXSelector,void* ptr)
{
    editor->setSelTextColor((FXColor)(FXuval)ptr);
    return 1;
}


// Update selected text foregoround color
long Preferences::onUpdTextSelForeColor(FXObject* sender,FXSelector,void*)
{
    FXColor color=editor->getSelTextColor();
    sender->handle(this,FXSEL(SEL_COMMAND,ID_SETINTVALUE),(void*)&color);
    return 1;
}


// Change selected text background color
long Preferences::onCmdTextSelBackColor(FXObject*,FXSelector,void* ptr)
{
    editor->setSelBackColor((FXColor)(FXuval)ptr);
    return 1;
}

// Update selected text background color
long Preferences::onUpdTextSelBackColor(FXObject* sender,FXSelector,void*)
{
    FXColor color=editor->getSelBackColor();
    sender->handle(this,FXSEL(SEL_COMMAND,ID_SETINTVALUE),(void*)&color);
    return 1;
}


// Change hilight text color
long Preferences::onCmdTextHiliteForeColor(FXObject*,FXSelector,void* ptr)
{
    editor->setHiliteTextColor((FXColor)(FXuval)ptr);
    return 1;
}

// Update hilight text color
long Preferences::onUpdTextHiliteForeColor(FXObject* sender,FXSelector,void*)
{
    FXColor color=editor->getHiliteTextColor();
    sender->handle(this,FXSEL(SEL_COMMAND,ID_SETINTVALUE),(void*)&color);
    return 1;
}


// Change hilight text background color
long Preferences::onCmdTextHiliteBackColor(FXObject*,FXSelector,void* ptr)
{
    editor->setHiliteBackColor((FXColor)(FXuval)ptr);
    return 1;
}

// Update hilight text background color
long Preferences::onUpdTextHiliteBackColor(FXObject* sender,FXSelector,void*)
{
    FXColor color=editor->getHiliteBackColor();
    sender->handle(this,FXSEL(SEL_COMMAND,ID_SETINTVALUE),(void*)&color);
    return 1;
}


// Change cursor color
long Preferences::onCmdTextCursorColor(FXObject*,FXSelector,void* ptr)
{
    editor->setCursorColor((FXColor)(FXuval)ptr);
    return 1;
}

// Update cursor color
long Preferences::onUpdTextCursorColor(FXObject* sender,FXSelector,void*)
{
    FXColor color=editor->getCursorColor();
    sender->handle(sender,FXSEL(SEL_COMMAND,FXWindow::ID_SETINTVALUE),(void*)&color);
    return 1;
}


// Change line numbers background color
long Preferences::onCmdTextBarColor(FXObject*,FXSelector,void* ptr)
{
    editor->setBarColor((FXColor)(FXuval)ptr);
    return 1;
}


// Update line numbers background color
long Preferences::onUpdTextBarColor(FXObject* sender,FXSelector,void*)
{
    FXColor color=editor->getBarColor();
    sender->handle(this,FXSEL(SEL_COMMAND,ID_SETINTVALUE),(void*)&color);
    return 1;
}


// Change line numbers color
long Preferences::onCmdTextNumberColor(FXObject*,FXSelector,void* ptr)
{
    editor->setNumberColor((FXColor)(FXuval)ptr);
    return 1;
}


// Update line numbers color
long Preferences::onUpdTextNumberColor(FXObject* sender,FXSelector,void*)
{
    FXColor color=editor->getNumberColor();
    sender->handle(this,FXSEL(SEL_COMMAND,ID_SETINTVALUE),(void*)&color);
    return 1;
}




// EditWindow class


// Map
FXDEFMAP(EditWindow) EditWindowMap[]=
{
	FXMAPFUNC(SEL_UPDATE,0,EditWindow::onUpdateTitle),
	FXMAPFUNC(SEL_FOCUSIN,0,EditWindow::onFocusIn),
	FXMAPFUNC(SEL_COMMAND,EditWindow::ID_ABOUT,EditWindow::onCmdAbout),
	FXMAPFUNC(SEL_COMMAND,EditWindow::ID_NEW,EditWindow::onCmdNew),
	FXMAPFUNC(SEL_COMMAND,EditWindow::ID_OPEN,EditWindow::onCmdOpen),
	FXMAPFUNC(SEL_COMMAND,EditWindow::ID_OPEN_RECENT,EditWindow::onCmdOpenRecent),
	FXMAPFUNC(SEL_COMMAND,EditWindow::ID_SAVE,EditWindow::onCmdSave),
	FXMAPFUNC(SEL_UPDATE,EditWindow::ID_SAVE,EditWindow::onUpdSave),
	FXMAPFUNC(SEL_COMMAND,EditWindow::ID_SAVEAS,EditWindow::onCmdSaveAs),
	FXMAPFUNC(SEL_COMMAND,EditWindow::ID_FONT,EditWindow::onCmdFont),
	FXMAPFUNC(SEL_COMMAND,EditWindow::ID_PRINT,EditWindow::onCmdPrint),
	FXMAPFUNC(SEL_UPDATE,EditWindow::ID_TOGGLE_WRAP,EditWindow::onUpdWrap),
	FXMAPFUNC(SEL_COMMAND,EditWindow::ID_TOGGLE_WRAP,EditWindow::onCmdWrap),
	FXMAPFUNC(SEL_COMMAND,EditWindow::ID_TOGGLE_LINES_NUM,EditWindow::onCmdLinesNum),
	FXMAPFUNC(SEL_UPDATE,EditWindow::ID_TOGGLE_LINES_NUM,EditWindow::onUpdLinesNum),
	FXMAPFUNC(SEL_INSERTED,EditWindow::ID_TEXT,EditWindow::onTextInserted),
	FXMAPFUNC(SEL_REPLACED,EditWindow::ID_TEXT,EditWindow::onTextReplaced),
	FXMAPFUNC(SEL_DELETED,EditWindow::ID_TEXT,EditWindow::onTextDeleted),
	FXMAPFUNC(SEL_RIGHTBUTTONRELEASE,EditWindow::ID_TEXT,EditWindow::onTextRightMouse),
	FXMAPFUNC(SEL_DND_MOTION,EditWindow::ID_TEXT,EditWindow::onEditDNDMotion),
	FXMAPFUNC(SEL_DND_DROP,EditWindow::ID_TEXT,EditWindow::onEditDNDDrop),
	FXMAPFUNC(SEL_UPDATE,EditWindow::ID_OVERSTRIKE,EditWindow::onUpdOverstrike),
	FXMAPFUNC(SEL_UPDATE,EditWindow::ID_NUM_ROWS,EditWindow::onUpdNumRows),
	FXMAPFUNC(SEL_COMMAND,EditWindow::ID_PREFERENCES,EditWindow::onCmdMorePrefs),
	FXMAPFUNC(SEL_COMMAND,EditWindow::ID_SEARCH,EditWindow::onCmdSearch),
	FXMAPFUNC(SEL_COMMAND,EditWindow::ID_REPLACE,EditWindow::onCmdReplace),
	FXMAPFUNC(SEL_COMMAND,EditWindow::ID_SEARCH_FORW_SEL,EditWindow::onCmdSearchSel),
	FXMAPFUNC(SEL_COMMAND,EditWindow::ID_SEARCH_BACK_SEL,EditWindow::onCmdSearchSel),
	FXMAPFUNC(SEL_COMMAND,EditWindow::ID_GOTO_LINE,EditWindow::onCmdGotoLine),
	FXMAPFUNCS(SEL_UPDATE, EditWindow::ID_WINDOW_1,EditWindow::ID_WINDOW_50,EditWindow::onUpdWindow),
	FXMAPFUNCS(SEL_COMMAND,EditWindow::ID_WINDOW_1,EditWindow::ID_WINDOW_50,EditWindow::onCmdWindow),
};


// Object implementation
FXIMPLEMENT(EditWindow,FXMainWindow,EditWindowMap,ARRAYNUMBER(EditWindowMap))

// Make some windows
EditWindow::EditWindow(XFileWrite* a,const FXString& file):FXMainWindow(a,"XFileWrite",NULL,NULL,DECOR_ALL,0,0,800,600,0,0),mrufiles(a)
{
    FXHotKey hotkey;
    FXButton *btn = NULL;

    // Add to list of windows
    getApp()->windowlist.append(this);

    // Default font
    font=NULL;

    // Application icons
    setIcon(xfwicon);

    // Status bar
    statusbar=new FXStatusBar(this,LAYOUT_SIDE_BOTTOM|LAYOUT_FILL_X|STATUSBAR_WITH_DRAGCORNER|FRAME_RAISED);

    // Make menu bar
    menubar=new FXMenuBar(this,LAYOUT_DOCK_NEXT|LAYOUT_SIDE_TOP|LAYOUT_FILL_X|FRAME_RAISED);

    // Sites where to dock
    FXDockSite* topdock=new FXDockSite(this,LAYOUT_SIDE_TOP|LAYOUT_FILL_X);
    FXDockSite* bottomdock=new FXDockSite(this,LAYOUT_SIDE_BOTTOM|LAYOUT_FILL_X);
    FXDockSite* leftdock=new FXDockSite(this,LAYOUT_SIDE_LEFT|LAYOUT_FILL_Y);
    FXDockSite* rightdock=new FXDockSite(this,LAYOUT_SIDE_RIGHT|LAYOUT_FILL_Y);

    // Toolbar
    dragshell=new FXToolBarShell(this,FRAME_RAISED);
    toolbar=new FXToolBar(topdock,dragshell,LAYOUT_DOCK_NEXT|LAYOUT_SIDE_TOP|LAYOUT_FILL_X|FRAME_RAISED);
    new FXToolBarGrip(toolbar,toolbar,FXToolBar::ID_TOOLBARGRIP,TOOLBARGRIP_DOUBLE);

    // File menu
    filemenu=new FXMenuPane(this);
    new FXMenuTitle(menubar,_("&File"),NULL,filemenu);

    // Edit Menu
    editmenu=new FXMenuPane(this);
    new FXMenuTitle(menubar,_("&Edit"),NULL,editmenu);

    // Search Menu
    searchmenu=new FXMenuPane(this);
    new FXMenuTitle(menubar,_("&Search"),NULL,searchmenu);

    // Preferences Menu
    prefsmenu=new FXMenuPane(this);
    new FXMenuTitle(menubar,_("&Preferences"),NULL,prefsmenu);

    // View menu
    viewmenu=new FXMenuPane(this);
    new FXMenuTitle(menubar,_("&View"),NULL,viewmenu);

    // Window menu
    windowmenu=new FXMenuPane(this);
    new FXMenuTitle(menubar,_("&Window"),NULL,windowmenu);

    // Help menu
    helpmenu=new FXMenuPane(this);
    new FXMenuTitle(menubar,_("&Help"),NULL,helpmenu,LAYOUT_LEFT);

    // Splitter
    FXSplitter* splitter=new FXSplitter(this,LAYOUT_SIDE_TOP|LAYOUT_FILL_X|LAYOUT_FILL_Y|SPLITTER_TRACKING);

    // Sunken border for text widget
    FXHorizontalFrame *textbox=new FXHorizontalFrame(splitter,FRAME_SUNKEN|FRAME_THICK|LAYOUT_FILL_X|LAYOUT_FILL_Y,0,0,0,0, 0,0,0,0);

    // Make editor window
    editor=new FXText(textbox,this,ID_TEXT,LAYOUT_FILL_X|LAYOUT_FILL_Y|TEXT_SHOWACTIVE);
    editor->setHiliteMatchTime(2000);
    editor->setTextStyle(editor->getTextStyle()|TEXT_FIXEDWRAP);
    editor->setTextStyle(editor->getTextStyle()&~TEXT_NO_TABS);
    editor->setScrollStyle(editor->getScrollStyle()&~SCROLLERS_DONT_TRACK);
    editor->setTextStyle(editor->getTextStyle()&~TEXT_SHOWACTIVE);
    editor->setMarginLeft(10);

    // Show insert mode in status bar
    FXLabel* overstrike=new FXLabel(statusbar,FXString::null,NULL,FRAME_SUNKEN|LAYOUT_RIGHT|LAYOUT_CENTER_Y,0,0,0,0,2,2,1,1);
    overstrike->setTarget(this);
    overstrike->setSelector(ID_OVERSTRIKE);

    // Show size of text in status bar
    FXTextField* numchars=new FXTextField(statusbar,7,this,ID_NUM_ROWS,TEXTFIELD_READONLY|FRAME_SUNKEN|JUSTIFY_RIGHT|LAYOUT_RIGHT|LAYOUT_CENTER_Y,0,0,0,0,2,2,1,1);
    numchars->setBackColor(statusbar->getBackColor());

    // Caption before number
    new FXLabel(statusbar,_("  Lines:"),NULL,LAYOUT_RIGHT|LAYOUT_CENTER_Y);

    // Show column number in status bar
    FXTextField* columnno=new FXTextField(statusbar,7,editor,FXText::ID_CURSOR_COLUMN,FRAME_SUNKEN|JUSTIFY_RIGHT|LAYOUT_RIGHT|LAYOUT_CENTER_Y,0,0,0,0,2,2,1,1);
    columnno->setBackColor(statusbar->getBackColor());

    // Caption before number
    new FXLabel(statusbar,_("  Col:"),NULL,LAYOUT_RIGHT|LAYOUT_CENTER_Y);

    // Show line number in status bar
    FXTextField* rowno=new FXTextField(statusbar,7,editor,FXText::ID_CURSOR_ROW,FRAME_SUNKEN|JUSTIFY_RIGHT|LAYOUT_RIGHT|LAYOUT_CENTER_Y,0,0,0,0,2,2,1,1);
    rowno->setBackColor(statusbar->getBackColor());

    // Caption before number
    new FXLabel(statusbar,_("  Line:"),NULL,LAYOUT_RIGHT|LAYOUT_CENTER_Y);

    // Toolbar buttons: File manipulation
    new FXButton(toolbar,_("\tNew (Ctrl-N)\tCreate new document. (Ctrl-N)"),newfileicon,this,ID_NEW,ICON_ABOVE_TEXT|BUTTON_TOOLBAR|FRAME_RAISED|LAYOUT_TOP|LAYOUT_LEFT);
    new FXButton(toolbar,_("\tOpen (Ctrl-O)\tOpen document file. (Ctrl-O)"),fileopenicon,this,ID_OPEN,ICON_ABOVE_TEXT|BUTTON_TOOLBAR|FRAME_RAISED|LAYOUT_TOP|LAYOUT_LEFT);
    new FXButton(toolbar,_("\tSave (Ctrl-S)\tSave document. (Ctrl-S)"),savefileicon,this,ID_SAVE,ICON_ABOVE_TEXT|BUTTON_TOOLBAR|FRAME_RAISED|LAYOUT_TOP|LAYOUT_LEFT);
    new FXButton(toolbar,_("\tClose (Ctrl-W)\tClose document file. (Ctrl-W)"),closefileicon,this,ID_CLOSE,ICON_ABOVE_TEXT|BUTTON_TOOLBAR|FRAME_RAISED|LAYOUT_TOP|LAYOUT_LEFT);

    // Spacer
    new FXSeparator(toolbar,SEPARATOR_GROOVE);

    // Toolbar buttons: Print
    new FXButton(toolbar,_("\tPrint (Ctrl-P)\tPrint document. (Ctrl-P)"),printicon,this,ID_PRINT,ICON_ABOVE_TEXT|BUTTON_TOOLBAR|FRAME_RAISED|LAYOUT_TOP|LAYOUT_LEFT);
    btn=new FXButton(toolbar,_("\tQuit (Ctrl-Q)\tQuit X File Write. (Ctrl-Q)"),quiticon,getApp(),XFileWrite::ID_CLOSEALL,ICON_ABOVE_TEXT|BUTTON_TOOLBAR|FRAME_RAISED|LAYOUT_TOP|LAYOUT_LEFT);
    btn->hide();
    hotkey=(CONTROLMASK<<16) | KEY_w;
    btn->addHotKey(hotkey);

    // Spacer
    new FXSeparator(toolbar,SEPARATOR_GROOVE);

    // Toolbar buttons: Editing
    new FXButton(toolbar,_("\tCopy (Ctrl-C)\tCopy selection to clipboard. (Ctrl-C)"),copy_clpicon,editor,FXText::ID_COPY_SEL,ICON_ABOVE_TEXT|BUTTON_TOOLBAR|FRAME_RAISED|LAYOUT_TOP|LAYOUT_LEFT);
    new FXButton(toolbar,_("\tCut (Ctrl-X)\tCut selection to clipboard. (Ctrl-X)"),cut_clpicon,editor,FXText::ID_CUT_SEL,ICON_ABOVE_TEXT|BUTTON_TOOLBAR|FRAME_RAISED|LAYOUT_TOP|LAYOUT_LEFT);
    new FXButton(toolbar,_("\tPaste (Ctrl-V)\tPaste clipboard. (Ctrl-V)"),paste_clpicon,editor,FXText::ID_PASTE_SEL,ICON_ABOVE_TEXT|BUTTON_TOOLBAR|FRAME_RAISED|LAYOUT_TOP|LAYOUT_LEFT);

    // Spacer
    new FXSeparator(toolbar,SEPARATOR_GROOVE);

    // Goto line
    new FXButton(toolbar,_("\tGoto line (Ctrl-L)\tGoto line number. (Ctrl-L)"),gotolineicon,this,EditWindow::ID_GOTO_LINE,ICON_ABOVE_TEXT|BUTTON_TOOLBAR|FRAME_RAISED|LAYOUT_TOP|LAYOUT_LEFT);

    // Spacer
    new FXSeparator(toolbar,SEPARATOR_GROOVE);

    // Undo/redo
    new FXButton(toolbar,_("\tUndo (Ctrl-Z)\tUndo last change. (Ctrl-Z)"),undoicon,&undolist,FXUndoList::ID_UNDO,ICON_ABOVE_TEXT|BUTTON_TOOLBAR|FRAME_RAISED|LAYOUT_TOP|LAYOUT_LEFT);
    new FXButton(toolbar,_("\tRedo (Ctrl-Y)\tRedo last undo. (Ctrl-Y)"),redoicon,&undolist,FXUndoList::ID_REDO,ICON_ABOVE_TEXT|BUTTON_TOOLBAR|FRAME_RAISED|LAYOUT_TOP|LAYOUT_LEFT);

    // Spacer
    new FXSeparator(toolbar,SEPARATOR_GROOVE);

    // Search
    new FXButton(toolbar,_("\tSearch (Ctrl-F)\tSearch text. (Ctrl-F)"),searchicon,this,EditWindow::ID_SEARCH,ICON_ABOVE_TEXT|BUTTON_TOOLBAR|FRAME_RAISED|LAYOUT_TOP|LAYOUT_LEFT);
    btn=new FXButton(toolbar,_("\tSearch selection backward (Shift-Ctrl-G, Shift-F3)\tSearch backward for selected text. (Shift-Ctrl-G, Shift-F3)"),searchprevicon,this,EditWindow::ID_SEARCH_BACK_SEL,ICON_ABOVE_TEXT|BUTTON_TOOLBAR|FRAME_RAISED|LAYOUT_TOP|LAYOUT_LEFT);
    hotkey=(0001<<16) | KEY_F3;
    btn->addHotKey(hotkey);
    btn=new FXButton(toolbar,_("\tSearch selection forward (Ctrl-G, F3)\tSearch forward for selected text. (Ctrl-G, F3)"),searchnexticon,this,EditWindow::ID_SEARCH_FORW_SEL,ICON_ABOVE_TEXT|BUTTON_TOOLBAR|FRAME_RAISED|LAYOUT_TOP|LAYOUT_LEFT);
    btn->addHotKey(KEY_F3);

    // Spacer
    new FXSeparator(toolbar,SEPARATOR_GROOVE);

    // Preferences
    new FXToggleButton(toolbar,_("\tWord wrap on (Ctrl-K)\tWord wrap on. (Ctrl-K)"),_("\tWord wrap off (Ctrl-K)\tWord wrap off. (Ctrl-K)"),wrapofficon,wraponicon,this,ID_TOGGLE_WRAP,BUTTON_TOOLBAR|LAYOUT_LEFT|ICON_BEFORE_TEXT);
    new FXToggleButton(toolbar,_("\tShow line numbers (Ctrl-T)\tShow line numbers. (Ctrl-T)"),_("\tHide line numbers (Ctrl-T)\tHide line numbers. (Ctrl-L)"),hidenumbersicon,shownumbersicon,this,ID_TOGGLE_LINES_NUM,BUTTON_TOOLBAR|LAYOUT_LEFT|ICON_BEFORE_TEXT);


    // File Menu entries
    new FXMenuCommand(filemenu,_("&New\tCtrl-N\tCreate new document. (Ctrl-N)"),newfileicon,this,ID_NEW);
    new FXMenuCommand(filemenu,_("&Open...\tCtrl-O\tOpen document file. (Ctrl-O)"),fileopenicon,this,ID_OPEN);
    new FXMenuCommand(filemenu,_("&Save\tCtrl-S\tSave changes to file. (Ctrl-S)"),savefileicon,this,ID_SAVE);
    new FXMenuCommand(filemenu,_("Save &As...\t\tSave document to another file."),saveasicon,this,ID_SAVEAS);
    new FXMenuCommand(filemenu,_("&Close\tCtrl-W\tClose document. (Ctrl-W)"),closefileicon,this,ID_CLOSE);
    new FXMenuSeparator(filemenu);
    new FXMenuCommand(filemenu,_("&Print...\tCtrl-P\tPrint document. (Ctrl-P)"),printicon,this,ID_PRINT);

    // Recent file menu; this automatically hides if there are no files
    FXMenuSeparator* sep1=new FXMenuSeparator(filemenu);
    sep1->setTarget(&mrufiles);
    sep1->setSelector(FXRecentFiles::ID_ANYFILES);
    new FXMenuCommand(filemenu,FXString::null,NULL,&mrufiles,FXRecentFiles::ID_FILE_1);
    new FXMenuCommand(filemenu,FXString::null,NULL,&mrufiles,FXRecentFiles::ID_FILE_2);
    new FXMenuCommand(filemenu,FXString::null,NULL,&mrufiles,FXRecentFiles::ID_FILE_3);
    new FXMenuCommand(filemenu,FXString::null,NULL,&mrufiles,FXRecentFiles::ID_FILE_4);
    new FXMenuCommand(filemenu,FXString::null,NULL,&mrufiles,FXRecentFiles::ID_FILE_5);
    new FXMenuCommand(filemenu,_("&Clear Recent Files"),NULL,&mrufiles,FXRecentFiles::ID_CLEAR);
    FXMenuSeparator* sep2=new FXMenuSeparator(filemenu);
    sep2->setTarget(&mrufiles);
    sep2->setSelector(FXRecentFiles::ID_ANYFILES);
    new FXMenuCommand(filemenu,_("&Quit\tCtrl-Q\tQuit X File Write. (Ctrl-Q)"),quiticon,getApp(),XFileWrite::ID_CLOSEALL);

    // Edit Menu entries
    new FXMenuCommand(editmenu,_("&Undo\tCtrl-Z\tUndo last change. (Ctrl-Z)"),undoicon,&undolist,FXUndoList::ID_UNDO);
    new FXMenuCommand(editmenu,_("&Redo\tCtrl-Y\tRedo last undo. (Ctrl-Y)"),redoicon,&undolist,FXUndoList::ID_REDO);
    new FXMenuCommand(editmenu,_("Revert to &saved\t\tRevert to saved."),reverticon,&undolist,FXUndoList::ID_REVERT);
    new FXMenuSeparator(editmenu);
    new FXMenuCommand(editmenu,_("&Copy\tCtrl-C\tCopy selection to clipboard. (Ctrl-C)"),copy_clpicon,editor,FXText::ID_COPY_SEL);
    new FXMenuCommand(editmenu,_("Cu&t\tCtrl-X\tCut selection to clipboard. (Ctrl-X)"),cut_clpicon,editor,FXText::ID_CUT_SEL);
    new FXMenuCommand(editmenu,_("&Paste\tCtrl-V\tPaste from clipboard. (Ctrl-V)"),paste_clpicon,editor,FXText::ID_PASTE_SEL);
    new FXMenuSeparator(editmenu);
    new FXMenuCommand(editmenu,_("Lo&wer-case\tCtrl-U\tChange to lower case. (Ctrl-U)"),lowercaseicon,editor,FXText::ID_LOWER_CASE);
    new FXMenuCommand(editmenu,_("Upp&er-case\tShift-Ctrl-U\tChange to upper case. (Shift-Ctrl-U)"),uppercaseicon,editor,FXText::ID_UPPER_CASE);
    new FXMenuCommand(editmenu,_("&Goto line...\tCtrl-L\tGoto line number. (Ctrl-L)"),gotolineicon,this,EditWindow::ID_GOTO_LINE);

    // Right mouse popup
    popupmenu=new FXMenuPane(this);
    new FXMenuCommand(popupmenu,_("&Undo"),undoicon,&undolist,FXUndoList::ID_UNDO);
    new FXMenuCommand(popupmenu,_("&Redo"),redoicon,&undolist,FXUndoList::ID_REDO);
    new FXMenuSeparator(popupmenu);
    new FXMenuCommand(popupmenu,_("&Copy"),copy_clpicon,editor,FXText::ID_COPY_SEL);
    new FXMenuCommand(popupmenu,_("Cu&t"),cut_clpicon,editor,FXText::ID_CUT_SEL);
    new FXMenuCommand(popupmenu,_("&Paste"),paste_clpicon,editor,FXText::ID_PASTE_SEL);
    new FXMenuCommand(popupmenu,_("Select &All"),NULL,editor,FXText::ID_SELECT_ALL);
    new FXMenuSeparator(popupmenu);

    // Search Menu entries
    new FXMenuCommand(searchmenu,_("&Search...\tCtrl-F\tSearch for a string. (Ctrl-F)"),searchicon,this,EditWindow::ID_SEARCH);
    new FXMenuCommand(searchmenu,_("&Replace...\tCtrl-R\tSearch for a string. (Ctrl-R)"),replaceicon,this,EditWindow::ID_REPLACE);
    new FXMenuCommand(searchmenu,_("Search sel. &backward\tShift-Ctrl-G\tSearch backward for selected text. (Shift-Ctrl-G, Shift-F3)"),searchprevicon,this,EditWindow::ID_SEARCH_BACK_SEL);
    new FXMenuCommand(searchmenu,_("Search sel. &forward\tCtrl-G\tSearch forward for selected text. (Ctrl-G, F3)"),searchnexticon,this,EditWindow::ID_SEARCH_FORW_SEL);


    // Preferences menu
    new FXMenuCheck(prefsmenu,_("&Word wrap\tCtrl-K\tToggle word wrap mode. (Ctrl-K)"),this,ID_TOGGLE_WRAP);
    new FXMenuCheck(prefsmenu,_("&Lines numbering\tCtrl-T\tToggle lines numbering mode. (Ctrl-T)"),this,ID_TOGGLE_LINES_NUM);
    new FXMenuCheck(prefsmenu,_("&Overstrike\t\tToggle overstrike mode."),editor,FXText::ID_TOGGLE_OVERSTRIKE);
    new FXMenuSeparator(prefsmenu);
    new FXMenuCommand(prefsmenu,_("&Font...\t\tChange text font."),fontsicon,this,ID_FONT);
    new FXMenuCommand(prefsmenu,_("&More preferences...\t\tChange other options."),prefsicon,this,ID_PREFERENCES);
    //new FXMenuSeparator(prefsmenu);
    //new FXMenuCommand(prefsmenu,_("&Save Settings\t\tSave settings now."),NULL,this,ID_SAVE_SETTINGS);

    // View Menu entries
    new FXMenuCheck(viewmenu,_("&Toolbar\t\tDisplay toolbar."),toolbar,FXWindow::ID_TOGGLESHOWN);
    new FXMenuCheck(viewmenu,_("&Status line\t\tDisplay status line."),statusbar,FXWindow::ID_TOGGLESHOWN);

    // Window menu
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_1);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_2);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_3);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_4);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_5);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_6);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_7);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_8);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_9);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_10);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_11);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_12);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_13);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_14);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_15);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_16);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_17);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_18);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_19);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_20);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_21);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_22);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_23);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_24);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_25);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_26);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_27);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_28);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_29);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_30);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_31);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_32);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_33);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_34);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_35);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_36);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_37);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_38);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_39);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_40);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_41);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_42);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_43);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_44);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_45);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_46);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_47);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_48);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_49);
    new FXMenuRadio(windowmenu,FXString::null,this,ID_WINDOW_50);

    // Help Menu entries
    new FXMenuCommand(helpmenu,_("&About X File Write...\tF1\tAbout X File Write. (F1)"),NULL,this,ID_ABOUT,0);

    // Dialogs
    printdialog=NULL;
    opendialog=NULL;
    savedialog=NULL;
    prefsdialog=NULL;
    searchdialog=NULL;
    replacedialog=NULL;

    // Add some alternative accelerators
    if(getAccelTable())
        getAccelTable()->addAccel(MKUINT(KEY_Z,CONTROLMASK|SHIFTMASK),&undolist,FXSEL(SEL_COMMAND,FXUndoList::ID_REDO));

    // Recent files
    mrufiles.setTarget(this);
    mrufiles.setSelector(ID_OPEN_RECENT);

    // Initialize file name
    filename=file;
    filenameset=FALSE;
    filetime=0;

    // Initialize other stuff
    stripcr=FALSE;
    linesnum=FALSE;
    undolist.mark();
}


// Create and show window
void EditWindow::create()
{
    loadConfig();
    FXMainWindow::create();
    dragshell->create();
    filemenu->create();
    editmenu->create();
    searchmenu->create();
    prefsmenu->create();
    viewmenu->create();
    windowmenu->create();
    helpmenu->create();
    popupmenu->create();
    if(!urilistType)
    {
        urilistType=getApp()->registerDragType(urilistTypeName);
    }
    show(PLACEMENT_DEFAULT);
    editor->setFocus();
}


// Detach window
void EditWindow::detach()
{
    FXMainWindow::detach();
    dragshell->detach();
    urilistType=0;
}


// Clean up
EditWindow::~EditWindow()
{
    getApp()->windowlist.remove(this);
    delete font;
    delete toolbar;
    delete menubar;
    delete dragshell;
    delete filemenu;
    delete editmenu;
    delete searchmenu;
    delete prefsmenu;
    delete viewmenu;
    delete windowmenu;
    delete helpmenu;
    delete popupmenu;
    delete editor;
    delete printdialog;
    delete opendialog;
    delete savedialog;
    delete prefsdialog;
    delete searchdialog;
    delete replacedialog;
}


// Is it modified
FXbool EditWindow::isModified() const
{
    return !undolist.marked();
}


// Load file
FXbool EditWindow::loadFile(const FXString& file)
{
    FXFile textfile(file,FXFile::Reading);
    FXint size,n,i,j,c;
    FXchar *text;

    // Opened file?
    if(!textfile.isOpen())
    {
        MessageBox::error(this,BOX_OK,_("Error Loading File"),_("Unable to open file: %s"),file.text());
        return FALSE;
    }

    // Get file size
    size=textfile.size();

    // Make buffer to load file
    if(!FXMALLOC(&text,FXchar,size))
    {
        MessageBox::error(this,BOX_OK,_("Error Loading File"),_("File is too big: %s (%d bytes)"),file.text(),size);
        return FALSE;
    }

    // Set wait cursor
    getApp()->beginWaitCursor();

    // Read the file
    n=textfile.readBlock(text,size);
    if(n<0)
    {
        FXFREE(&text);
        MessageBox::error(this,BOX_OK,_("Error Loading File"),_("Unable to read file: %s"),file.text());
        getApp()->endWaitCursor();
        return FALSE;
    }

    // Strip carriage returns
    if(stripcr)
    {
        for(i=j=0; j<n; j++)
        {
            c=text[j];
            if(c!='\r')
                text[i++]=c;
        }
        n=i;
    }

    // Set text
    editor->setText(text,n);
    FXFREE(&text);

    // Lines numbering
    if (linesnum)
    {
        FXuint size=editor->getNumRows();
        FXuint cols=(FXuint)ceil(log10(size));
        editor->setBarColumns(cols);
    }
    else
        editor->setBarColumns(0);

    // Kill wait cursor
    getApp()->endWaitCursor();

    mrufiles.appendFile(file);
    filetime=FXStat::modified(file);
    filenameset=TRUE;
    filename=file;

    // Clear undo records
    undolist.clear();

    // Mark undo state as clean (saved)
    undolist.mark();

    return TRUE;
}




// Save file
FXbool EditWindow::saveFile(const FXString& file)
{
    FXFile textfile(file,FXFile::Writing);
    FXint size,n;
    FXchar *text;

    // Opened file?
    if(!textfile.isOpen())
    {
        MessageBox::error(this,BOX_OK,_("Error Saving File"),_("Unable to open file: %s"),file.text());
        return FALSE;
    }

    // Get size
    size=editor->getLength();

    // Alloc buffer
    if(!FXMALLOC(&text,FXchar,size+1))
    {
        MessageBox::error(this,BOX_OK,_("Error Saving File"),_("File is too big: %s"),file.text());
        return FALSE;
    }

    // Set wait cursor
    getApp()->beginWaitCursor();

    // Get text from editor
    editor->getText(text,size);

    // Write the file
    n=textfile.writeBlock(text,size);

    // Ditch buffer
    FXFREE(&text);

    // Kill wait cursor
    getApp()->endWaitCursor();

    // Were we able to write it all?
    if(n!=size)
    {
        MessageBox::error(this,BOX_OK,_("Error Saving File"),_("File: %s truncated."),file.text());
        return FALSE;
    }

    mrufiles.appendFile(file);
    filetime=FXStat::modified(file);
    filenameset=TRUE;
    filename=file;
    undolist.mark();
    return TRUE;
}


// Generate unique name for a new window
FXString EditWindow::unique() const
{
    FXString name=_("untitled");
    for(FXint i=1; i<2147483647; i++)
    {
        if(!findWindow(name)) break;
        name.format(_("untitled%d"),i);
    }
    return name;
}


// Find an as yet untitled, unedited window
EditWindow *EditWindow::findUnused() const
{
    for(FXint w=0; w<getApp()->windowlist.no(); w++)
    {
        if(!getApp()->windowlist[w]->isFilenameSet() && !getApp()->windowlist[w]->isModified())
        {
            return getApp()->windowlist[w];
        }
    }
    return NULL;
}


// Find window, if any, currently editing the given file
EditWindow *EditWindow::findWindow(const FXString& file) const
{
    for(FXint w=0; w<getApp()->windowlist.no(); w++)
    {
        if(getApp()->windowlist[w]->getFilename()==file)
            return getApp()->windowlist[w];
    }
    return NULL;
}


// Visit given line
void EditWindow::visitLine(FXint line)
{
    FXint pos=editor->nextLine(0,line-1);
    editor->setCursorPos(pos);
    editor->setCenterLine(pos);
}


// Read configuration from registry
void EditWindow::loadConfig()
{
    FXColor textback,textfore,textselback,textselfore,textcursor,texthilitefore,texthiliteback;
    FXColor textbar,textnumber;
    FXint ww,hh,xx,yy,wrapping,wrapcols,tabcols;
    FXint hidestatus,hidetoolbar,hilitematchtime;
    FXString fontspec;

    // Text colors
    textback=getApp()->reg().readColorEntry("SETTINGS","textbackground",editor->getBackColor());
    textfore=getApp()->reg().readColorEntry("SETTINGS","textforeground",editor->getTextColor());
    textselback=getApp()->reg().readColorEntry("SETTINGS","textselbackground",editor->getSelBackColor());
    textselfore=getApp()->reg().readColorEntry("SETTINGS","textselforeground",editor->getSelTextColor());
    textcursor=getApp()->reg().readColorEntry("SETTINGS","textcursor",editor->getCursorColor());
    texthiliteback=getApp()->reg().readColorEntry("SETTINGS","texthilitebackground",editor->getHiliteBackColor());
    texthilitefore=getApp()->reg().readColorEntry("SETTINGS","texthiliteforeground",editor->getHiliteTextColor());
    textbar=getApp()->reg().readColorEntry("SETTINGS","textnumberbackground",editor->getBarColor());
    textnumber=getApp()->reg().readColorEntry("SETTINGS","textnumberforeground",editor->getNumberColor());

    // Font
    fontspec=getApp()->reg().readStringEntry("SETTINGS","textfont","");
    if(!fontspec.empty())
    {
        font=new FXFont(getApp(),fontspec);
        font->create();
        editor->setFont(font);
    }

    // Get size
    ww=getApp()->reg().readUnsignedEntry("SETTINGS","width",800);
    hh=getApp()->reg().readUnsignedEntry("SETTINGS","height",600);
    xx=getApp()->reg().readIntEntry("SETTINGS","xpos",5);
    yy=getApp()->reg().readIntEntry("SETTINGS","ypos",5);

    // Showing the status line?
    hidestatus=getApp()->reg().readIntEntry("SETTINGS","hidestatus",FALSE);

    // Showing the tool bar?
    hidetoolbar=getApp()->reg().readIntEntry("SETTINGS","hidetoolbar",FALSE);

    // Highlight match time
    hilitematchtime=getApp()->reg().readIntEntry("SETTINGS","hilitematchtime",3000);

    // Word wrapping
    wrapping=getApp()->reg().readIntEntry("SETTINGS","wordwrap",0);
    wrapcols=getApp()->reg().readIntEntry("SETTINGS","wrapcols",80);

    // Tab settings
    tabcols=getApp()->reg().readIntEntry("SETTINGS","tabcols",8);

    // Various flags
    stripcr=getApp()->reg().readIntEntry("SETTINGS","stripreturn",FALSE);
    linesnum=getApp()->reg().readIntEntry("SETTINGS","linesnum",FALSE);

    // Change the colors
    editor->setTextColor(textfore);
    editor->setBackColor(textback);
    editor->setSelBackColor(textselback);
    editor->setSelTextColor(textselfore);
    editor->setCursorColor(textcursor);
    editor->setHiliteBackColor(texthiliteback);
    editor->setHiliteTextColor(texthilitefore);
    editor->setBarColor(textbar);
    editor->setNumberColor(textnumber);


    // Hide statusline
    if(hidestatus) statusbar->hide();

    // Hide toolbar
    if(hidetoolbar) toolbar->hide();

    // Wrap mode
    if(wrapping)
        editor->setTextStyle(editor->getTextStyle()|TEXT_WORDWRAP);
    else
        editor->setTextStyle(editor->getTextStyle()&~TEXT_WORDWRAP);

    // Wrap and tab columns
	editor->setWrapColumns(wrapcols);
    editor->setTabColumns(tabcols);

    // Highlight match time
    editor->setHiliteMatchTime(hilitematchtime);

    // Reposition window
    position(xx,yy,ww,hh);
}


// Save configuration to registry
void EditWindow::saveConfig()
{
    FXString fontspec;

    // Colors of text
    getApp()->reg().writeColorEntry("SETTINGS","textbackground",editor->getBackColor());
    getApp()->reg().writeColorEntry("SETTINGS","textforeground",editor->getTextColor());
    getApp()->reg().writeColorEntry("SETTINGS","textselbackground",editor->getSelBackColor());
    getApp()->reg().writeColorEntry("SETTINGS","textselforeground",editor->getSelTextColor());
    getApp()->reg().writeColorEntry("SETTINGS","textcursor",editor->getCursorColor());
    getApp()->reg().writeColorEntry("SETTINGS","texthilitebackground",editor->getHiliteBackColor());
    getApp()->reg().writeColorEntry("SETTINGS","texthiliteforeground",editor->getHiliteTextColor());
    getApp()->reg().writeColorEntry("SETTINGS","textnumberbackground",editor->getBarColor());
    getApp()->reg().writeColorEntry("SETTINGS","textnumberforeground",editor->getNumberColor());


    // Write new window size back to registry
    getApp()->reg().writeUnsignedEntry("SETTINGS","width",getWidth());
    getApp()->reg().writeUnsignedEntry("SETTINGS","height",getHeight());
    getApp()->reg().writeIntEntry("SETTINGS","xpos",getX());
    getApp()->reg().writeIntEntry("SETTINGS","ypos",getY());

    // Was status line shown
    getApp()->reg().writeIntEntry("SETTINGS","hidestatus",!statusbar->shown());

    // Was toolbar shown
    getApp()->reg().writeIntEntry("SETTINGS","hidetoolbar",!toolbar->shown());

    // Highlight match time
    getApp()->reg().writeIntEntry("SETTINGS","hilitematchtime",editor->getHiliteMatchTime());

    // Wrap mode
    getApp()->reg().writeIntEntry("SETTINGS","wordwrap",(editor->getTextStyle()&TEXT_WORDWRAP)!=0);
	getApp()->reg().writeIntEntry("SETTINGS","wrapcols",editor->getWrapColumns());

    // Tab settings
    getApp()->reg().writeIntEntry("SETTINGS","tabcols",editor->getTabColumns());

    // Strip returns
    getApp()->reg().writeIntEntry("SETTINGS","stripreturn",stripcr);
    getApp()->reg().writeIntEntry("SETTINGS","linesnum",linesnum);

    // Search path
    getApp()->reg().writeStringEntry("SETTINGS","searchpath",searchpath.text());

    // Font
    fontspec=editor->getFont()->getFont();
    getApp()->reg().writeStringEntry("SETTINGS","textfont",fontspec.text());
}


// About box
long EditWindow::onCmdAbout(FXObject*,FXSelector,void*)
{
    FXString msg=_("X File Write Version ");
    msg = msg + VERSION + _(" is a simple text editor.\n\nCopyright (C) 2002-2007 Roland Baudin (roland65@free.fr)");
    MessageBox about(this,_("About X File Write"),msg.text(),xfwicon,BOX_OK|DECOR_TITLE|DECOR_BORDER);
    about.execute(PLACEMENT_OWNER);
    return 1;
}

// Show preferences dialog
long EditWindow::onCmdMorePrefs(FXObject*,FXSelector,void*)
{
    if (prefsdialog==NULL)
        prefsdialog=new Preferences(this);
    prefsdialog->execute(PLACEMENT_OWNER);
    return 1;
}


// Change text font
long EditWindow::onCmdFont(FXObject*,FXSelector,void*)
{
    FontDialog fontdlg(this,_("Change Font"),DECOR_BORDER|DECOR_TITLE);
    FXFontDesc fontdesc;
    editor->getFont()->getFontDesc(fontdesc);
    fontdlg.setFontSelection(fontdesc);
    if(fontdlg.execute())
    {
        FXFont *oldfont=font;
        fontdlg.getFontSelection(fontdesc);
        font=new FXFont(getApp(),fontdesc);
        font->create();
        editor->setFont(font);
        delete oldfont;
    }
    saveConfig();
    return 1;
}


// New
long EditWindow::onCmdNew(FXObject*,FXSelector,void*)
{
    EditWindow *window=new EditWindow(getApp(),unique());
    window->create();
    window->raise();
    window->setFocus();
    return 1;
}


// Open
long EditWindow::onCmdOpen(FXObject*,FXSelector,void*)
{
    const FXchar *patterns[]=
        {
            _("All Files"),          "*",
            _("Text Files"),         "*.txt",
            _("C Source Files"),     "*.c",
            _("C++ Source Files"),   "*.cpp",
            _("C++ Source Files"),   "*.cc",
            _("C++ Source Files"),   "*.cxx",
            _("C/C++ Header Files"), "*.h",
            _("HTML Files"),         "*.html",
            _("HTML Files"),         "*.htm",
            _("PHP Files"),          "*.php",NULL
        };
    if (opendialog==NULL)
        opendialog=new FileDialog(this,_("Open Document"),0,0,0,650,480);
    opendialog->setSelectMode(SELECTFILE_MULTIPLE);
    opendialog->setPatternList(patterns);
    opendialog->setDirectory(FXPath::directory(filename));
    if(opendialog->execute())
    {
        FXString* files=opendialog->getFilenames();
        FXuint i=0;
        while (files[i]!=FXString::null)
        {
            EditWindow *window=findWindow(files[i]);
            if(!window)
            {
                window=findUnused();
                if(!window)
                {
                    window=new EditWindow(getApp(),unique());
                    window->create();
                }
                window->loadFile(files[i]);
            }
            window->raise();
            window->setFocus();
            i++;
        }
        delete[] files;
    }
    return 1;
}


// Open recent file
long EditWindow::onCmdOpenRecent(FXObject*,FXSelector,void* ptr)
{
    FXString file=(const char*)ptr;
    EditWindow *window=findWindow(file);
    if(!window)
    {
        window=findUnused();
        if(!window)
        {
            window=new EditWindow(getApp(),unique());
            window->create();
        }
        window->loadFile(file);
    }
    window->raise();
    window->setFocus();
    return 1;
}


// See if we can get it as a filename
long EditWindow::onEditDNDDrop(FXObject*,FXSelector,void*)
{
    FXchar *data;
    FXuint len;
    if(getDNDData(FROM_DRAGNDROP,urilistType,(FXuchar*&)data,len))
    {
        FXString urilist(data,len);
        FXString file=FXURL::fileFromURL(urilist.before('\r'));
        FXFREE(&data);
        if(file.empty())
            return 1;
        if(!saveChanges())
            return 1;
        loadFile(file);
        return 1;
    }
    return 0;
}


// See if a filename is being dragged over the window
long EditWindow::onEditDNDMotion(FXObject*,FXSelector,void*)
{
    if(offeredDNDType(FROM_DRAGNDROP,urilistType))
    {
        acceptDrop(DRAG_COPY);
        return 1;
    }
    return 0;
}


// Save changes, prompt for new filename
FXbool EditWindow::saveChanges()
{
    FXuint answer;
    FXString file;
    if(isModified())
    {
        answer=MessageBox::question(this,BOX_YES_NO_CANCEL,_("Unsaved Document"),_("Save %s to file?"),FXPath::name(filename).text());
        if(answer==BOX_CLICKED_CANCEL) return FALSE;
        if(answer==BOX_CLICKED_YES)
        {
            file=filename;
            if(!filenameset)
            {
                if (savedialog==NULL)
                    savedialog=new FileDialog(this,_("Save Document"),0,0,0,650,480);
                savedialog->setSelectMode(SELECTFILE_ANY);
                savedialog->setFilename(file);
                if(!savedialog->execute()) return FALSE;
                file=savedialog->getFilename();
                if(FXStat::exists(file))
                    if(BOX_CLICKED_NO==MessageBox::question(this,BOX_YES_NO,_("Overwrite Document"),_("Overwrite existing document: %s?"),file.text())) return FALSE;
            }
            saveFile(file);
        }
    }
    return TRUE;
}


// Save
long EditWindow::onCmdSave(FXObject* sender,FXSelector sel,void* ptr)
{
    if(!filenameset) return onCmdSaveAs(sender,sel,ptr);
    saveFile(filename);
    return 1;
}


// Save Update
long EditWindow::onUpdSave(FXObject* sender,FXSelector,void*)
{
    sender->handle(this,isModified()?FXSEL(SEL_COMMAND,ID_ENABLE):FXSEL(SEL_COMMAND,ID_DISABLE),NULL);
    return 1;
}


// Save As
long EditWindow::onCmdSaveAs(FXObject*,FXSelector,void*)
{
    if (savedialog==NULL)
        savedialog=new FileDialog(this,_("Save Document"),0,0,0,650,480);
    FXString file=filename;
    savedialog->setSelectMode(SELECTFILE_ANY);
    savedialog->setFilename(file);
    if(savedialog->execute())
    {
        file=savedialog->getFilename();
        if(FXStat::exists(file))
        {
            if(BOX_CLICKED_NO==MessageBox::question(this,BOX_YES_NO,_("Overwrite Document"),_("Overwrite existing document: %s?"),file.text()))
                return 1;
        }
        saveFile(file);
    }
    return 1;
}


// Close window
FXbool EditWindow::close(FXbool notify)
{
    // Prompt to save changes
    if(!saveChanges())
        return FALSE;

    // Save settings
    saveConfig();

    // Perform normal close stuff
    return FXMainWindow::close(notify);
}


// User clicks on one of the window menus
long EditWindow::onCmdWindow(FXObject*,FXSelector sel,void*)
{
    FXint which=FXSELID(sel)-ID_WINDOW_1;
    if(which<getApp()->windowlist.no())
    {
        getApp()->windowlist[which]->raise();
        getApp()->windowlist[which]->setFocus();
    }
    return 1;
}


// Update handler for window menus
long EditWindow::onUpdWindow(FXObject *sender,FXSelector sel,void*)
{
    FXint which=FXSELID(sel)-ID_WINDOW_1;
    if(which<getApp()->windowlist.no())
    {
        EditWindow *window=getApp()->windowlist[which];
        FXString string;
        if(which<49)
            string.format("&%d %s",which+1,window->getTitle().text());
        else
            string.format("5&0 %s",window->getTitle().text());

        sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_SETSTRINGVALUE),(void*)&string);

        if(window==getApp()->getActiveWindow())
            sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_CHECK),NULL);

        else
            sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_UNCHECK),NULL);
        sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_SHOW),NULL);
    }
    else
        sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_HIDE),NULL);
    return 1;
}


// Update title from current filename
long EditWindow::onUpdateTitle(FXObject* sender,FXSelector sel,void* ptr)
{
    FXMainWindow::onUpdate(sender,sel,ptr);
    FXString title=FXPath::name(getFilename());
    if(isModified())
        title+=_(" (changed)");
    FXString directory=FXPath::directory(getFilename());
    if(!directory.empty())
        title+=" - " + directory;
    setTitle(title);
    return 1;
}


// Print the text
long EditWindow::onCmdPrint(FXObject*,FXSelector,void*)
{
    // Read the current print command from the registry
    FXString printcommand, command;
    printcommand=getApp()->reg().readStringEntry("SETTINGS","print_command","lpr -P printer");

    // Open print dialog filled with the current print command
    int rc=1;
    if (printdialog==NULL)
        printdialog=new InputDialog(this,printcommand,_("Print command: \n(ex: lpr -P <printer>)"),_("Print"),"",printbigicon);
    printdialog->setText(printcommand);
    printdialog->CursorEnd();
    rc=printdialog->execute(PLACEMENT_CURSOR);
    printcommand=printdialog->getText();

    // If cancel was pressed, exit
    if (!rc)
        return 0;

    // Write the new print command to the registry
    getApp()->reg().writeStringEntry("SETTINGS","print_command",printcommand.text());

    // Perform the print command
    command =  "cat " + filename + " |" + printcommand + " &";
    system(command.text());

    return 1;
}


// Toggle wrap mode
long EditWindow::onCmdWrap(FXObject*,FXSelector,void*)
{
    editor->setTextStyle(editor->getTextStyle()^TEXT_WORDWRAP);
    return 1;
}


// Update toggle wrap mode
long EditWindow::onUpdWrap(FXObject* sender,FXSelector,void*)
{
    if(editor->getTextStyle()&TEXT_WORDWRAP)
        sender->handle(this,FXSEL(SEL_COMMAND,ID_CHECK),NULL);
    else
        sender->handle(this,FXSEL(SEL_COMMAND,ID_UNCHECK),NULL);
    return 1;
}

// Toggle lines numbering
long EditWindow::onCmdLinesNum(FXObject*,FXSelector,void*)
{
    linesnum=!linesnum;
    if (linesnum)
    {
        FXuint size=editor->getNumRows();
        FXuint cols=(FXuint)ceil(log10(size));
        editor->setBarColumns(cols);
    }
    else
        editor->setBarColumns(0);

    return 1;
}


// Update toggle lines number
long EditWindow::onUpdLinesNum(FXObject* sender,FXSelector,void*)
{
    if(linesnum)
        sender->handle(this,FXSEL(SEL_COMMAND,ID_CHECK),NULL);
    else
        sender->handle(this,FXSEL(SEL_COMMAND,ID_UNCHECK),NULL);
    return 1;
}


// Update box for overstrike mode display
long EditWindow::onUpdOverstrike(FXObject* sender,FXSelector,void*)
{
    FXString mode((editor->getTextStyle()&TEXT_OVERSTRIKE)?_("OVR"):_("INS"));
    sender->handle(this,FXSEL(SEL_COMMAND,ID_SETSTRINGVALUE),(void*)&mode);
    return 1;
}


// Update box for size display
long EditWindow::onUpdNumRows(FXObject* sender,FXSelector,void*)
{
    FXuint size=editor->getNumRows();
    sender->handle(this,FXSEL(SEL_COMMAND,ID_SETINTVALUE),(void*)&size);
    return 1;
}




// Text inserted; callback has [pos nins]
long EditWindow::onTextInserted(FXObject*,FXSelector,void* ptr)
{
    const FXTextChange *change=(const FXTextChange*)ptr;

    // Log undo record
    if(!undolist.busy())
    {
        undolist.add(new FXTextInsert(editor,change->pos,change->nins,change->ins));
        if(undolist.size()>MAXUNDOSIZE) undolist.trimSize(KEEPUNDOSIZE);
    }


    return 1;
}


// Text deleted; callback has [pos ndel]
long EditWindow::onTextDeleted(FXObject*,FXSelector,void* ptr)
{
    const FXTextChange *change=(const FXTextChange*)ptr;

    // Log undo record
    if(!undolist.busy())
    {
        undolist.add(new FXTextDelete(editor,change->pos,change->ndel,change->del));
        if(undolist.size()>MAXUNDOSIZE) undolist.trimSize(KEEPUNDOSIZE);
    }

    return 1;
}


// Text replaced; callback has [pos ndel nins]
long EditWindow::onTextReplaced(FXObject*,FXSelector,void* ptr)
{
    const FXTextChange *change=(const FXTextChange*)ptr;

    // Log undo record
    if(!undolist.busy())
    {
        undolist.add(new FXTextReplace(editor,change->pos,change->ndel,change->nins,change->del,change->ins));
        if(undolist.size()>MAXUNDOSIZE) undolist.trimSize(KEEPUNDOSIZE);
    }

    return 1;
}


// Released right button
long EditWindow::onTextRightMouse(FXObject*,FXSelector,void* ptr)
{
    FXEvent* event=(FXEvent*)ptr;
    if(!event->moved)
    {
        allowPopupScroll=TRUE; // Allow keyboard scrolling
        popupmenu->popup(NULL,event->root_x,event->root_y);
        getApp()->runModalWhileShown(popupmenu);
        allowPopupScroll=FALSE;
    }
    return 1;
}


// Check file when focus moves in
long EditWindow::onFocusIn(FXObject* sender,FXSelector sel,void* ptr)
{
    register FXTime t;
    FXMainWindow::onFocusIn(sender,sel,ptr);
    if(filetime!=0)
    {
        t=FXStat::modified(filename);
        if(t && t!=filetime)
        {
            filetime=t;
            if(BOX_CLICKED_OK==MessageBox::warning(this,BOX_OK_CANCEL,_("File Was Changed"),_("%s\nwas changed by another program. Reload this file from disk?"),filename.text()))
            {
                FXint top=editor->getTopLine();
                FXint pos=editor->getCursorPos();
                loadFile(filename);
                editor->setTopLine(top);
                editor->setCursorPos(pos);
            }
        }
    }
    return 1;
}


// Search text
long EditWindow::onCmdSearch(FXObject*,FXSelector,void*)
{
    if (searchdialog==NULL)
        searchdialog=new FXSearchDialog(this,_("Search"),searchicon);
    FXint beg[10];
    FXint end[10];
    FXint pos;
    FXuint code;
    FXString searchstring;
    FXuint searchflags;
    do
    {
        code=searchdialog->execute();
        if(code==FXSearchDialog::DONE)
            return 1;
        searchstring=searchdialog->getSearchText();
        searchflags=searchdialog->getSearchMode();
        pos=editor->isPosSelected(editor->getCursorPos()) ? (searchflags&SEARCH_BACKWARD) ? editor->getSelStartPos()-1 : editor->getSelEndPos() : editor->getCursorPos();
        if(editor->findText(searchstring,beg,end,pos,searchflags|SEARCH_WRAP,10))
        {
            editor->setAnchorPos(beg[0]);
            editor->extendSelection(end[0],SELECT_CHARS,TRUE);
            editor->setCursorPos(end[0],TRUE);
            editor->makePositionVisible(beg[0]);
            editor->makePositionVisible(end[0]);
        }
        else
        {
            getApp()->beep();
        }
    }
    while(code==FXSearchDialog::SEARCH_NEXT);
    return 1;
}


// Replace text; we assume that findText has called squeezegap()!
long EditWindow::onCmdReplace(FXObject*,FXSelector,void*)
{
    replacedialog=new FXReplaceDialog(this,_("Replace"),replaceicon);
    FXint beg[10],end[10],fm,to,len,pos;
    FXuint searchflags,code;
    FXString searchstring;
    FXString replacestring;
    FXString replacevalue;

    // Get text into buffer
    FXchar* buffer;
    FXint length=editor->getLength();
    FXMALLOC(&buffer,FXchar,length);
    editor->getText(buffer,length);

    do
    {
        code=replacedialog->execute();
        if(code==FXReplaceDialog::DONE)
            goto ret;
        searchflags=replacedialog->getSearchMode();
        searchstring=replacedialog->getSearchText();
        replacestring=replacedialog->getReplaceText();
        replacevalue=FXString::null;
        fm=-1;
        to=-1;
        if(code==FXReplaceDialog::REPLACE_ALL)
        {
            searchflags&=~SEARCH_BACKWARD;
            pos=0;
            while(editor->findText(searchstring,beg,end,pos,searchflags,10))
            {
                if(0<=fm)
                    replacevalue.append(&buffer[pos],beg[0]-pos);
                replacevalue.append(FXRex::substitute(buffer,length,beg,end,replacestring,10));
                if(fm<0)
                    fm=beg[0];
                to=end[0];
                pos=end[0];
                if(beg[0]==end[0])
                    pos++;
            }
        }
        else
        {
            pos=editor->isPosSelected(editor->getCursorPos()) ? (searchflags&SEARCH_BACKWARD) ? editor->getSelStartPos()-1 : editor->getSelEndPos() : editor->getCursorPos();
            if(editor->findText(searchstring,beg,end,pos,searchflags|SEARCH_WRAP,10))
            {
                replacevalue=FXRex::substitute(buffer,length,beg,end,replacestring,10);
                fm=beg[0];
                to=end[0];
            }
        }
        if(0<=fm)
        {
            len=replacevalue.length();
            editor->replaceText(fm,to-fm,replacevalue.text(),len,TRUE);
            editor->setCursorPos(fm+len,TRUE);
            editor->makePositionVisible(editor->getCursorPos());
            editor->setModified(TRUE);
        }
        else
        {
            getApp()->beep();
        }
    }
    while(code==FXReplaceDialog::REPLACE_NEXT);

ret:
    FXFREE(&buffer);
    return 1;
}



// Search for selected text
long EditWindow::onCmdSearchSel(FXObject*,FXSelector sel,void*)
{
    FXString string;
    FXuint searchflags;
    FXString searchstring;
    FXint pos=editor->getCursorPos();
    FXint beg,end;

    // First, try UTF-8
    if(getDNDData(FROM_SELECTION,utf8Type,string))
    {
        FXTRACE((100,"Search UTF8\n"));
        searchstring=string;
    }

    // Next, try UTF-16
    else if(getDNDData(FROM_SELECTION,utf16Type,string))
    {
        FXTRACE((100,"Search UTF16\n"));
        FXUTF16LECodec unicode;                 // FIXME maybe other endianness for unix
        searchstring=unicode.mb2utf(string);
    }

    // Finally, try good old 8859-1
    else if(getDNDData(FROM_SELECTION,stringType,string))
    {
        FXTRACE((100,"Search ASCII\n"));
        FX88591Codec ascii;
        searchstring=ascii.mb2utf(string);
    }

    // No dice!
    else
    {
        goto x;
    }
	
    // Search direction
    if(FXSELID(sel)==ID_SEARCH_FORW_SEL)
    {
        if(editor->isPosSelected(pos)) pos=editor->getSelEndPos();
        searchflags=SEARCH_EXACT|SEARCH_FORWARD;
    }
    else
    {
        if(editor->isPosSelected(pos)) pos=editor->getSelStartPos()-1;
        searchflags=SEARCH_EXACT|SEARCH_BACKWARD;
    }

    // Perform search
    if(editor->findText(searchstring,&beg,&end,pos,searchflags|SEARCH_WRAP))
    {
        if(beg!=editor->getSelStartPos() || end!=editor->getSelEndPos())
        {
            editor->setAnchorPos(beg);
            editor->extendSelection(end,SELECT_CHARS,TRUE);
            editor->setCursorPos(end);
            editor->makePositionVisible(beg);
            editor->makePositionVisible(end);
            return 1;
        }
    }

    // Beep
x:
    getApp()->beep();
    return 1;
}


// Go to line number
long EditWindow::onCmdGotoLine(FXObject*,FXSelector,void*)
{
    FXint row=editor->getCursorRow()+1;
    if(FXInputDialog::getInteger(row,this,_("Goto Line"),_("&Goto line number:"),gotobigicon,1,2147483647))
    {
        update();
        editor->setCursorRow(row-1,TRUE);
        editor->makePositionVisible(editor->getCursorPos());
    }
    return 1;
}
