// *************************************************************************
// * Xgsm - mobile phone manager
// *
// * File:    xgsm_single_editor.h
// *
// * Purpose: A single list of phonebook or SMS entries that can be edited
// *
// * Author:  Peter Hofmann (software@pxh.de)
// *
// * Created: 8.3.2000
// *************************************************************************

#ifndef XGSM_SINGLE_EDITOR_H
#define XGSM_SINGLE_EDITOR_H

#include "xgsm_device.h"
#include "xgsm_dialogs.h"
#include "xgsm_multi_editor.h"

#include <gtk--/frame.h>
#include <gtk--/clist.h>
#include <gtk--/optionmenu.h>
#include <gtk--/menuitem.h>
#include <gnome--/entry.h>
#include <sigc++/signal_system.h>
#include <gsmlib/gsm_util.h>
#include <set>

namespace Xgsm
{
  // forward declarations
  class MultiEditor;
  class PbSingleEditor;
  class SMSSingleEditor;

  // single editor frame
  class SingleEditor : public DeviceHelper
  {
  protected:
    // widgets
    MultiEditor *_multiEditor;
    Gtk::Frame *_frame;
    
    // widgets that must be insensitive if no device is open
    // or the device is busy
    Gtk::CList *_clist;
    Gtk::MenuItem *_openFileMI, *_openDeviceMI, *_closeMI, *_undoMI;
    Gtk::MenuItem *_redoMI, *_cutMI, *_copyMI, *_pasteMI, *_selectAllMI;
    Gtk::MenuItem *_unselectAllMI, *_addMI;
    Gtk::Button *_addButton, *_cutButton, *_resetButton, *_openButton;
    Gnome::Entry *_storeNameEntry;
    Gtk::OptionMenu *_sourceMenu;
    set<string> _storeNameHistory; // mirrors history of _storeNameEntry
    int _sourceMenuSetting;
    bool _storeNameActivateEvent; // onStoreNameActivate() called

    gsmlib::SortOrder _sortOrder; // clist sort order
    string _prefix;             // prefix for widget names (eg. "pb_left")
    string _extension;          // file name extension (.pb, .sms)
    EntryListRef _entries;      // entry cache (from onRefresh())
    int _rowEdited;             // if == -1 no edit in progress
    bool _deviceIdle;           // last setting with ::sensitive()

    RequestRef _openRequest;    // pending open request
    string _deviceName;         // device name
    SigC::Connection _idle;     // connection to the idle() function

    SingleEditor(MultiEditor *multiEditor,
                 Gtk::Frame *frame,
                 string prefix, string messageText, string extension,
                 gsmlib::SortOrder sortOrder);

    // refresh editor pane
    virtual void refresh() = 0;

    // start edit of row
    void startEdit(int row);

    // stop edit in progress
    void stopEdit();

    // return true if edit in progress
    bool editInProgress() const {return _rowEdited != -1;}

    // if edit is in progress, update store with new contents of edited entry
    virtual void completeEdit(bool editorClosed = false) = 0;
    void onCompleteEdit() {completeEdit(false);}

    // set or reset sensitivity of buttons and entry fields
    // depending on parameter
    virtual void sensitive(bool deviceIdle);

    // idle function, used to wait for device to finish
    gint idle();

    // device callbacks
    void onAddDone(ResponseRef response);
    void onCutDone(ResponseRef response);
    void onOpenDone(ResponseRef response);
    void onEditDone(ResponseRef response);
    void onCloseDone(ResponseRef response);

    // gui callbacks
    void onDestroy();
    void onClose();
    void onSourceMenu();
    void onOpen();
    void onStoreNameActivate();
    void onOpenFile();
    void onOpenDevice();

    // return number of rows currently selected
    int rowsSelected();

    // return appropriate store name vector from OpenResponse
    virtual vector<string> getStoreNames(OpenResponse &res) = 0;

    // return current store name
    virtual string currentStoreName() = 0;

    // debugging function
    virtual void dumpEntries() = 0;

  public:
    // functions to connect editor to device
    virtual void open(string deviceName, string baudRate,
                      string initString, bool swHandshake,
                      string phoneBook) = 0;
    virtual void open(string fileName) = 0;

    // initialize phonebook editor (must be called if no open())
    virtual void init() = 0;

    // copy and synchronize operations
    virtual void copyStore(int operation, gsmlib::Ref<SingleEditor> e) = 0;
    
    // return device (for use by MultiEditor)
    DeviceRef device() {return _dev;}

    // close device
    void close() {onClose();}

    // only used for SMSSingleEditor
    virtual void setPanedPosition(int position);
    
    virtual ~SingleEditor();

    friend class PbSingleEditor;
    friend class SMSSingleEditor;
  };
};

#endif // XGSM_SINGLE_EDITOR_H
