// *************************************************************************
// * Xgsm - mobile phone manager
// *
// * File:    xgsm_sms_single_editor.cc
// *
// * Purpose: A single list of SMS store entries that can be edited
// *
// * Author:  Peter Hofmann (software@pxh.de)
// *
// * Created: 28.10.2000
// *************************************************************************

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "xgsm_sms_single_editor.h"
#include "xgsm_multi_editor.h"
#include "xgsm_util.h"
#include "xgsm_dialogs.h"
#include "xgsm_pref.h"

#include <gtk--/base.h>
#include <gtk--/frame.h>
#include <gtk--/button.h>
#include <gtk--/main.h>
#include <gtk--/paned.h>
#include <gtk--/box.h>
#include <gnome--/dialog.h>
#include <typeinfo>
#include <algorithm>
#include <gsmlib/gsm_util.h>
#include <limits.h>
#include <iostream>

extern "C" {
#include "interface.h"
#include "support.h"
}

using namespace std;
using namespace Xgsm;
using namespace SigC;
using namespace gsmlib;

// SMSSingleEditor members

void SMSSingleEditor::sensitive(bool deviceIdle)
{
  _messageEntry->set_sensitive(deviceIdle && ! _dev.isnull());
  _numberEntry->set_sensitive(deviceIdle && ! _dev.isnull());
  SingleEditor::sensitive(deviceIdle);
}

void SMSSingleEditor::refresh()
{
  clearText(_messageEntry);
  _numberEntry->set_text("");
  if (_dev.isnull())
  {
    _clist->clear();
    ((Gtk::Entry*)_storeNameEntry->gtk_entry())->set_text("");
  }
  else
    _completionConnection =
      _dev->request(
        new GetSMSEntriesRequest(_smsStore, _sortOrder,
                                 slot(this, &SMSSingleEditor::onRefresh)));
}

void SMSSingleEditor::completeEdit(bool editorClosed)
{
  if (_rowEdited == -1) return;
  stopEdit();
}

void SMSSingleEditor::startEdit(int row)
{
  SingleEditor::startEdit(row);
  _messageEntry->set_editable(false);
  _numberEntry->set_editable(false);
}

void SMSSingleEditor::stopEdit()
{
  SingleEditor::stopEdit();
  _messageEntry->set_editable(true);
  _numberEntry->set_editable(true);
}

void SMSSingleEditor::onRefresh(ResponseRef response)
{
  if (typeid(response()) == typeid(ErrorResponse))
  {
    // handle error response
    Gnome::Dialogs::error
      (stringPrintf(_("Error reading %s \n("), _messageText.c_str()) +
       ((ErrorResponse&)response()).exception().what() + ")")->run_and_close();
    close();
  }
  else
  {
    assert(typeid(response()) == typeid(GetSMSEntriesResponse));
    if (_destroyed) return;
    _entries = ((GetSMSEntriesResponse&)response()).entries();
    _clist->freeze();
    _clist->clear();
    for (EntryList::iterator i = _entries->begin(); i != _entries->end(); ++i)
    {
      vector<string> newRow;
      newRow.push_back(stringPrintf("%d", i->smsIndex()));
      newRow.push_back(i->smsType());
      newRow.push_back(i->smsDate());
      newRow.push_back(i->smsTelephone());
      _clist->append_row(Gtk::SArray(newRow));
    }
    _clist->thaw();
    stopEdit();
    sensitive(_deviceIdle);
  }
}

void SMSSingleEditor::onTextChanged()
{
  string text = _messageEntry->get_chars(0, -1);
  if (text.length() > _maxTextLength)
  {
    text = text.substr(0, _maxTextLength);
    unsigned int savePoint = _messageEntry->get_point();
    if (savePoint > text.length() - 1)
      savePoint = text.length() - 1;
    clearText(_messageEntry, false);
    _messageEntry->insert(text);
    _messageEntry->set_point(savePoint);
    _messageEntry->thaw();
  }
}

void SMSSingleEditor::onNumberChanged()
{
  string number = _numberEntry->get_text();
  if (number.length() > _maxTelephoneLength)
  {
    number = number.substr(0, _maxTelephoneLength);
    _numberEntry->set_text(number);
  }
}

void SMSSingleEditor::onVboxAllocate(GtkAllocation *allocation)
{
  if (! _recursiveVboxAllocation)
  {
    // height of the upper vbox is the position to remember
    config.setSmsPanedPosition(allocation->height);
    
    // tell the other vbox
    if (_multiEditor != NULL)
      _multiEditor->setPanedPosition(this, allocation->height);
  }
  _recursiveVboxAllocation = false;
}

void SMSSingleEditor::onSelectRow(int row, int col, GdkEvent *event)
{
  //  completeEdit();
  if (rowsSelected() == 1)
  {
    clearText(_messageEntry, false);
    _messageEntry->insert(_entries()[row].smsText());
    _messageEntry->thaw();
    _numberEntry->set_text(_entries()[row].smsTelephone());
    startEdit(row);
  }
  else
  {
    stopEdit();
    clearText(_messageEntry);
    _numberEntry->set_text("");
  }
  sensitive(true);
}

void SMSSingleEditor::onUnselectRow(int row, int col, GdkEvent *event)
{
  //  completeEdit();
  if ((rowsSelected() == 2 && _rowEdited == row) || rowsSelected() == 1)
  {
    int r = _clist->selection().begin()->get_row_num();
    startEdit(r);
    clearText(_messageEntry, false);
    _messageEntry->insert(_entries()[row].smsText());
    _messageEntry->thaw();
    _numberEntry->set_text(_entries()[row].smsTelephone());
  }
  else
  {
    stopEdit();
    clearText(_messageEntry);
    _numberEntry->set_text("");
  }
  sensitive(true);
}

void SMSSingleEditor::onExtendSelection()
{
  // FIXME, remove
}

void SMSSingleEditor::onSelectAll()
{
  // FIXME, remove
}

void SMSSingleEditor::onUnselectAll()
{
  // FIXME, remove
}

void SMSSingleEditor::onCut()
{
  // stop edit in progress, since the entry is deleted anyway
  stopEdit();

  DeleteSmsEntriesRequest *req =
    new DeleteSmsEntriesRequest(_smsStore,
                                slot(this, &SingleEditor::onCutDone));
  for (Gtk::CList_Helpers::SelectionIterator i = _clist->selection().begin();
       i != _clist->selection().end(); ++i)
  {
    DEBUG_START(1);
    cout << "cutting " << i->get_row_num() << endl;
    DEBUG_END;
    req->addEntry(_entries()[i->get_row_num()]);
  }
  _completionConnection = _dev->request(req);
}

void SMSSingleEditor::onAdd()
{
  EntryListRef e = new EntryList;
  SMSStoreEntry newEntry(new SMSSubmitMessage(_messageEntry->get_chars(0, -1),
                                              _numberEntry->get_text()));
  e->push_back(newEntry);
  _completionConnection =
    _dev->request(
      new AddSmsEntryRequest(AddSmsEntryRequest::Add, _smsStore, e,
                             slot(this, &SingleEditor::onAddDone)));
}

void SMSSingleEditor::onColumnClicked(gint column)
{
  SortOrder oldSortOrder = _sortOrder;
  switch (column)
  {
  case 0:
    _sortOrder = ByIndex;
    break;
  case 1:
    _sortOrder = ByType;
    break;
  case 2:
    _sortOrder = ByDate;
    break;
  case 3:
    _sortOrder = ByAddress;
    break;
  }
  if (_sortOrder != oldSortOrder)
    refresh();
}

#ifndef NDEBUG

void SMSSingleEditor::dumpEntries()
{
  DEBUG_START(1);
  /*for (set<string>::iterator i = _storeNameHistory.begin();
       i != _storeNameHistory.end(); ++i)
       cout << "history: " << *i << endl;*/
  DEBUG_END;
}
#endif

SMSSingleEditor::SMSSingleEditor(MultiEditor *multiEditor, 
                                 Gtk::Frame *frame, bool isLeft) :
  SingleEditor(multiEditor, 
               frame, isLeft ? "sms_left" : "sms_right",
               isLeft ? _("left SMS store") : _("right SMS store"),
               SMSSTORE_FILE_EXTENSION, ByDate),
  _recursiveVboxAllocation(false),
  _maxTextLength(UINT_MAX), _maxTelephoneLength(UINT_MAX)
{
  // connect inherited widgets
  _clist->select_row.connect(slot(this, &SMSSingleEditor::onSelectRow));
  _clist->unselect_row.connect(slot(this, &SMSSingleEditor::onUnselectRow));
  _clist->click_column.connect(slot(this, &SMSSingleEditor::onColumnClicked));
  _addButton->clicked.connect(slot(this, &SMSSingleEditor::onAdd));
  _cutButton->clicked.connect(slot(this, &SMSSingleEditor::onCut));
  _addMI->activate.connect(slot(this, &SMSSingleEditor::onAdd));
  _cutMI->activate.connect(slot(this, &SMSSingleEditor::onCut));

  // lookup and connect own widgets
  LOOKUP_WIDGET(_messageEntry, _prefix + "_message_entry", GtkText,
                GTK_IS_TEXT, frame);
  _messageEntry->changed.connect(slot(this, &SMSSingleEditor::onTextChanged));
  _messageEntry->set_word_wrap(true);
  LOOKUP_WIDGET(_numberEntry, _prefix + "_number_entry", GtkEntry,
                GTK_IS_ENTRY, frame);
  _numberEntry->changed.connect(slot(this, &SMSSingleEditor::onNumberChanged));

  // notify about size changes of upper vbox of paned object...
  Gtk::VBox *vbox;
  LOOKUP_WIDGET(vbox, _prefix + "_vbox", GtkVBox, GTK_IS_VBOX, frame);
  vbox->size_allocate.connect(slot(this, &SMSSingleEditor::onVboxAllocate));

  LOOKUP_WIDGET(_paned, _prefix + "_paned", GtkPaned, GTK_IS_PANED, frame);
  
  // ...so that they can be remembered
  int position;
  if (config.getSmsPanedPosition(position))
    _paned->set_position(position);
}

void SMSSingleEditor::open(string deviceName, string baudRate, 
                           string initString, bool swHandshake,
                           string smsStore)
{
  close();
  _smsStore = smsStore;
  _sourceName = _deviceName = deviceName;
  DeviceHelper::open(new OpenRequest(deviceName, baudRate,
                                     initString, swHandshake,
                                     slot(this, &DeviceHelper::onDeviceEvent),
                                     slot(this, &SingleEditor::onOpenDone)),
                     true);
}

void SMSSingleEditor::open(string fileName)
{
  close();
  _smsStore = fileName;
  _sourceName = fileName;
  _deviceName = config.getPhoneDevice();
  DeviceHelper::open(new OpenRequest(fileName, SMSFile,
                                     slot(this, &DeviceHelper::onDeviceEvent),
                                     slot(this, &SingleEditor::onOpenDone)),
                     true);
}

void SMSSingleEditor::init()
{
  _deviceName = config.getPhoneDevice();
  // just find out store names
  DeviceHelper::open(new OpenRequest("", SMSFile,
                                     slot(this, &DeviceHelper::onDeviceEvent),
                                     slot(this, &SingleEditor::onOpenDone)),
                     false);
}

void SMSSingleEditor::copyStore(int operation, SingleEditorRef e)
{
  AddSmsEntryRequest::Operation op;
  switch (operation)
  {
  case CopySelected:
    op = AddSmsEntryRequest::Add;
    break;
  case CopyAll:
    op = AddSmsEntryRequest::Add;
    break;
  case Synchronize:
    op = AddSmsEntryRequest::Synchronize;
    break;
  case Backup:
    op = AddSmsEntryRequest::Backup;
    break;
  }

  EntryListRef copyList = new EntryList;

  if (operation == CopySelected)
    for (Gtk::CList_Helpers::SelectionIterator i =
           e->_clist->selection().begin();
         i != e->_clist->selection().end(); ++i)
      copyList->push_back(e->_entries()[i->get_row_num()]);
  else
    copyList() = e->_entries(); // deep copy
  
  _completionConnection =
    _dev->request(
      new AddSmsEntryRequest(op, _smsStore, copyList,
                             slot(this, &SingleEditor::onAddDone)));
}

void SMSSingleEditor::setPanedPosition(int position)
{
  if (! _recursiveVboxAllocation)
  {
    _recursiveVboxAllocation = true;
    _paned->set_position(position);
  }
}
