/*
 * Copyright (C) 2003 the xmms-kde team
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
#ifndef __XMMS_KDEDB_H__
#define __XMMS_KDEDB_H__

#include <string>

#if defined __GNUC__ && (__GNUC__ > 2)
  #include <ext/hash_set>
  using namespace __gnu_cxx;
#else
  #include <hash_set>
#endif

#include <kconfig.h>

#include <qhbuttongroup.h>
#include <qevent.h>
#include <qprogressbar.h>
#include <qfiledialog.h>
#include <qwidget.h>
#include <qstring.h>
#include <qstringlist.h>
#include <qvbox.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <qlistbox.h>
#include <qpoint.h>
#include <qpixmap.h>
#include <qsize.h>
#include <qradiobutton.h>
#include <qthread.h>
#include <qvbuttongroup.h>

#include "sqlite.h"

#include "playerinterface.h"

using namespace std;

class StatusLabel;
class ProgressLabel;

class InsertThread;
class SearchThread;

class XmmsKdeDB : public QObject {

  Q_OBJECT

public:
  XmmsKdeDB(KConfig *conf, QPixmap *icon);

  ~XmmsKdeDB();

  bool connectDB();

  void disconnectDB();

  bool isConnectedDB();

  bool isEnabled() {
    return enable;
  }

  void disableDB() {
    enable = false;
    disconnectDB();
  }

  void enableDB() {
    enable = true;
  }

  sqlite *db;
  sqlite *querydb;

  QWidget* getConfigurationWidget(QWidget *parent);
  void writeConfig(); 

  bool getSync() {

    return sync;
  }

signals:
  
  void statusChanged(QString status);

public slots:

  void stopInsertThread();
  void updateDatabase();

private slots:

  void addPathToList();
  void removePathFromList();

  void configurationChanged();

private:

  void readConfig();

  bool enable; 
  bool connected;

  QString name; 

  KConfig *config;

  QCheckBox *enableBox;
  QListBox *pathListBox;
  QStringList pathList;
  QFrame *statusFrame;
  QPixmap *icon;
  InsertThread *insertThread;
  bool sync;
};


class XmmsKdeDBQuery : public QVBox {

  Q_OBJECT

public:

  XmmsKdeDBQuery(XmmsKdeDB *datab, PlayerInterface *p, 
		 QPixmap *icon, KConfig *conf);

  ~XmmsKdeDBQuery();

  void setPlayer(PlayerInterface *p);
  void popup();

  QWidget* getConfigurationWidget(QWidget *parent);
  void writeConfig();

  int getPopup() {

    return pop;
  }
  
  void customEvent(QCustomEvent *e);

signals:
  
public slots:

private slots:

  void newQuery();

  void play(int index);
  
  void setPlayList();
  void addPlayList();

  void popupChanged(int index);

private:

  void readConfig();

  XmmsKdeDB *db;
  PlayerInterface *player;

  QListBox *resultBox;
  QLineEdit *query;

  QHButtonGroup *buttons;

  KConfig *config;

  QVButtonGroup *queryGroup;
  QPoint framePos;
  QSize frameSize;
  int pop;

  SearchThread *searchThread;
};

class QueryItem : public QListBoxText {


public:
  QueryItem(QString text, QString file);
  ~QueryItem();

  QString getFile();


private:

  QString filename;

};


//! A class to insert items into a database in a new thread.
/*!
  InsertThread is used to update the contents of the database. 
  A new thread is created to allow the GUI to be updated as this operation
  goes on.
  InsertThread reads the contents of the database and compares them to the 
  given pathnames. When there are differences between the entries in the 
  database and the filesystem, the tags of the files are scanned and inserted
  into the database.
*/
class InsertThread : public QThread {

 public:
  //! create a new InsertThread.
  /*!
    \param database       the sqlite database.
    \param directories    the pathnames to process.
    \param statusLabel    the status label in the GUI to update accordingly.
    \param fileLabel      the label in the GUI that shows the current filename.
    \param progressLabel  the progressbar in the GUI.
  */
  InsertThread(sqlite *database, 
	       QStringList directories,
	       StatusLabel *statusLabel,
	       StatusLabel *fileLabel, 
	       ProgressLabel *progressLabel);

  //! the main method of the thread.
  /*! 
    The database is updated to contain the same entries as the processed
    paths.
  */
  virtual void run();

 private:

  //! update the database
  /*!
    First, the database entries are read into a hashtable. Then, the filenames
    in the given paths are read into another hashtable.
    Then, the files that are no longer in the filesystem but in the database
    are deleted from the database.
    At last, the files that are in the filesystem but not in the database
    are added.
    \param dir the pathnames to process.
  */
  void updateDatabase(QStringList dir);

  //! inserts the tag of a file into the database.
  /*!
    \param file    the file to add to the database.
  */
  void insertIntoDatabase(QString file);

  //! delete a file entry from the database.
  /*!
    \param file  the file to delete.
  */
  void deleteFromDatabase(QString file);

  //! build a vector with the filenames of all .ogg and .mp3 files in the path.
  /*!
    \param path    the path to process.
    \param vector  the vector to build from the path.
  */
  void addPathToVector(QString path, vector<QString *> *vector);

  //! the sqlite database object.
  sqlite *db;
  
  //! the directories to add to the database.
  QStringList dir;

  //! the label which shows the status of this InsertThread.
  StatusLabel *statusLabel;

  //! the label which shows the file that is currently processed.
  StatusLabel *fileLabel;

  //! the progessbar.
  ProgressLabel *progressLabel;
};

class ResultEvent: public QCustomEvent {

 public:
  ResultEvent(QStringList m): QCustomEvent(60041), s(m) {}

  QStringList message() const {
    return s;
  }
 private:
  QStringList s;
};

//! A class to search items in a database
class SearchThread : public QThread {

 public:
  //! create a new SearchThread.
  /*!
    \param database       the sqlite database.
  */
  SearchThread(sqlite *database, XmmsKdeDBQuery *q, QString what, QString like);

  //! the main method of the thread.
  virtual void run();

 private:

  //! the sqlite database object.
  sqlite *db;

  //! the query dialog
  XmmsKdeDBQuery *query;

  QString what, like;
};  



class StatusEvent: public QCustomEvent {

 public:
  StatusEvent(QString m): QCustomEvent(60042), s(m) {}

  QString message() const {
    return s;
  }
 private:
  QString s;
};

class ProgressStepEvent: public QCustomEvent {

 public:
  ProgressStepEvent(int p): QCustomEvent(60043) {
    prog = p;
  }

  int progress() const {
    return prog;
  }
 private:
  int prog;
};

class ProgressTotalEvent: public QCustomEvent {

 public:
  ProgressTotalEvent(int p): QCustomEvent(60044) {
    prog = p;
  }

  int progress() const {
    return prog;
  }
 private:
  int prog;
};

class StatusLabel: public QLabel {

 public:
  StatusLabel(char *c, QWidget *parent, int len): QLabel(c, parent) {
    length = len;
  }

  void customEvent(QCustomEvent *e) {
    if (e->type() == 60042) {
      StatusEvent *s = (StatusEvent *) e;
      QString st = s->message();
      st.truncate(length);
      setText(st);
    }
  }

 private:
  int length;

};

class ProgressLabel: public QProgressBar {

 public:
  ProgressLabel(QWidget *parent): QProgressBar(0, parent) {

    //setTotalSteps(0);
  }

  void customEvent(QCustomEvent *e) {
    if (e->type() == 60043) {
      // progress step
      ProgressStepEvent *s = (ProgressStepEvent *) e;
      setProgress(s->progress());

    } else if (e->type() == 60044) {
      // progress total
      ProgressTotalEvent *s = (ProgressTotalEvent *) e;
      setTotalSteps(s->progress());
    }
  }

 private:
  int length;

};
#endif



