/*
 * Copyright (C) 2001,  Espen Skoglund <esk@ira.uka.de>
 * Copyright (C) 2001, 2004  Haavard Kvaalen <havardk@xmms.org>
 * Copyright (C) 2004,  Matti Hmlinen <ccr@tnsp.org>
 *                
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 *                
 */

#if defined(HAVE_CONFIG_H)
#include "config.h"
#endif

#include <glib.h>
#include <stdio.h>
#include <string.h>

#include "titlestring.h"
#include "../xmms/i18n.h"

struct padding {
	int side, width, precision;
	char padchar;
};

enum {
	PAD_SIDE_LEFT,
	PAD_SIDE_RIGHT,
};


static int xmms_vputstr(GString *output, char *pstr, struct padding *pad)
{
	int i;
	/* Lenght of the string that is actually printed */
	int plen;
	
	if (!pstr)
		return FALSE;

	/* Calculate printed size */
	plen = strlen(pstr);
	if (pad->precision >= 0 && pad->precision < plen)
		plen = pad->precision;

	/* Do left-padding */
	if (pad->width > 0)
	{
		if (pad->side == PAD_SIDE_LEFT)
		{
			for (i = pad->width - plen; i > 0; i--)
				g_string_append_c(output, pad->padchar);
		}
	}

	/* Insert the string */
	if (pad->precision >= 0)
	{
		for (i = 0; i < plen; i++)
			g_string_append_c(output, *pstr++);
	}
	else
		g_string_append(output, pstr);
	
	/* Do right-padding */
	if (pad->side == PAD_SIDE_RIGHT && pad->width > 0)
		for (i = pad->width - plen; i > 0; i--)
			g_string_append_c(output, ' ');
	return TRUE;
}


static int xmms_vputnum(GString *output, int ival, struct padding *pad)
{
	int i, n, ndigits;
	char *pstr;

	/* FIXME: Never print the value '0'? */
	if (ival == 0)
		return FALSE;
	
	/* Create string */
	pstr = g_strdup_printf("%d", ival);
	ndigits = strlen(pstr);

	n = MAX(ndigits, pad->precision);

	/* Do left-padding */
	if (pad->side == PAD_SIDE_LEFT && pad->width > n)
	{
		for (i = pad->width - n; i > 0; i--)
			g_string_append_c(output, pad->padchar);
	}

	/* Do zero-padding (precision) */
	for (i = n - ndigits; i-- > 0;)
		g_string_append_c(output, '0');
	
	/* Add the value */
	g_string_append(output, pstr);
	g_free(pstr);

	/* Do right-padding */
	if (pad->side == PAD_SIDE_RIGHT && pad->width > 0)
		for (i = pad->width - n; i > 0; i--)
			g_string_append_c(output, ' ');
	return TRUE;
}


#define VISDIGIT(c) ((c >= '0') && (c <= '9'))

#define VCHECK(input, field) \
	(((char *) &input->field - (char *) input) < input->__size)

#define VS(field) (VCHECK(input, field) ? input->field : NULL)
#define VI(field) (VCHECK(input, field) ? input->field : 0)

#define VPUTSTR(pval) xmms_vputstr(string, VS(pval), &padding)

#define VPUTNUM(pval) xmms_vputnum(string, VI(pval), &padding)


char *xmms_get_titlestring(char *fmt, TitleInput *input)
{
	gboolean f_output;
	GString *string;
	char *strptr;
	struct padding padding;

	if (!fmt)
		return NULL;

	f_output = FALSE;
	string = g_string_new("");

	while (*fmt)
	switch (*fmt)
	{
		case '%':
			fmt++;

			padding.side = PAD_SIDE_LEFT;
			padding.width = -1;
			padding.precision = -1;
			padding.padchar = ' ';

			/* Check for right padding */
			if (*fmt == '-')
			{
				padding.side = PAD_SIDE_RIGHT;
				fmt++;
			}
			/* Parse field width */
			if (VISDIGIT(*fmt))
			{
				if (*fmt == '0')
				{
					padding.padchar = '0';
					fmt++;
				}
				padding.width = 0;
				while (VISDIGIT(*fmt))
				{
					padding.width *= 10;
					padding.width += *fmt - '0';
					fmt++;
				}
			}

			/* Parse precision */
			if (*fmt == '.')
			{
				fmt++;
				if (VISDIGIT(*fmt))
				{
					padding.precision = 0;
					while (VISDIGIT(*fmt))
					{
						padding.precision *= 10;
						padding.precision += *fmt - '0';
						fmt++;
					}
				}
			}

			/* Parse format conversion */
			switch (*fmt++)
			{
				case 'a':
					f_output += VPUTSTR(album_name);
					break;

				case 'c':
					f_output += VPUTSTR(comment);
					break;

				case 'd':
					f_output += VPUTSTR(date);
					break;

				case 'e':
					f_output += VPUTSTR(file_ext);
					break;

				case 'f':
					f_output += VPUTSTR(file_name);
					break;

				case 'F':
					f_output += VPUTSTR(file_path);
					break;

				case 'g':
					f_output += VPUTSTR(genre);
					break;

				case 'n':
					f_output += VPUTNUM(track_number);
					break;

				case 'p':
					f_output += VPUTSTR(performer);
					break;

				case 't':
					f_output += VPUTSTR(track_name);
					break;

				case 'y':
					f_output += VPUTNUM(year);
					break;

				case '%':
					g_string_append_c(string, '%');
					break;

				default:
					g_string_append_c(string, '%');
					if (*fmt)
						g_string_append_c(string, *fmt);
					break;
			}
			break;

		default:
			/* Normal character */
			g_string_append_c(string, *fmt++);
			break;
	}

	/* Check if we actually output anything */
	if (!f_output)
	{
		g_string_free(string, TRUE);
		return NULL;
	}

	/* Return the result */
	strptr = string->str;
	g_string_free(string, FALSE);
	return strptr;
}



static struct tagdescr {
	char tag;
	char *description;
} descriptions[] = {
	{'p', N_("Performer/Artist")},
	{'a', N_("Album")},
	{'g', N_("Genre")},
	{'f', N_("File name")},
	{'F', N_("File path")},
	{'e', N_("File extension")},
	{'t', N_("Track name")},
	{'n', N_("Track number")},
	{'d', N_("Date")},
	{'y', N_("Year")},
	{'c', N_("Comment")},
};

GtkWidget* xmms_titlestring_descriptions(char* tags, int columns)
{
	GtkWidget *table, *label;
	gchar tagstr[5];
	gint num = strlen(tags);
	int r, c, i;

	g_return_val_if_fail(tags != NULL, NULL);
	g_return_val_if_fail(columns <= num, NULL);

	table = gtk_table_new((num + columns - 1) / columns, columns * 2, FALSE);
	gtk_table_set_row_spacings(GTK_TABLE(table), 2);
	gtk_table_set_col_spacings(GTK_TABLE(table), 5);

	for (c = 0; c < columns; c++)
	{
		for (r = 0; r < (num + columns - 1 - c) / columns; r++)
		{
			sprintf(tagstr, "%%%c:", *tags);
			label = gtk_label_new(tagstr);
			gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
			gtk_table_attach(GTK_TABLE(table), label, 2 * c, 2 * c + 1, r, r + 1,
					 GTK_FILL, GTK_FILL, 0, 0);
			gtk_widget_show(label);

			for (i = 0; i <= sizeof(descriptions) / sizeof(struct tagdescr); i++)
			{
				if (i == sizeof(descriptions) / sizeof(struct tagdescr))
					g_warning("xmms_titlestring_descriptions(): Invalid tag: %c", *tags);
				else if (*tags == descriptions[i].tag)
				{
					label = gtk_label_new(gettext(descriptions[i].description));
					gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
					gtk_table_attach(GTK_TABLE(table), label, 2 * c + 1, 2 * c + 2, r, r + 1,
							 GTK_EXPAND | GTK_FILL,  GTK_EXPAND | GTK_FILL, 0 ,0);
					gtk_widget_show(label);
					break;
				}
			}
			tags++;
		}
	}

	return table;
}

