// file      : xsde/cxx/serializer/validator.cxx
// author    : Boris Kolpackov <boris@codesynthesis.com>
// copyright : Copyright (c) 2005-2007 Code Synthesis Tools CC
// license   : GNU GPL v2 + exceptions; see accompanying LICENSE file

#include <cxx/serializer/validator.hxx>

#include <xsd-frontend/semantic-graph.hxx>
#include <xsd-frontend/traversal.hxx>

#include <cxx/serializer/elements.hxx>

#include <iostream>

using std::wcerr;

namespace CXX
{
  namespace Serializer
  {
    namespace
    {
      class ValidationContext: public Context
      {
      public:
        ValidationContext (SemanticGraph::Schema& root,
                           CLI::Options const& options,
                           Boolean& valid_)
            : Context (std::wcerr, root, options),
              valid (valid_)
        {
        }

      public:
        String
        xpath (SemanticGraph::Nameable& n)
        {
          if (n.is_a<SemanticGraph::Namespace> ())
            return L"<namespace-level>"; // There is a bug if you see this.

          assert (n.named ());

          SemanticGraph::Scope& scope (n.scope ());

          if (scope.is_a<SemanticGraph::Namespace> ())
            return n.name ();

          return xpath (scope) + L"/" + n.name ();
        }

      protected:
        ValidationContext (ValidationContext& c)
            :  Context (c),
               valid (c.valid)
        {
        }

      protected:
        Boolean& valid;
      };

      //
      //
      struct Traverser : Traversal::Schema,
                         Traversal::Complex,
                         Traversal::Type,
                         protected virtual ValidationContext
      {
        Traverser (ValidationContext& c)
            : ValidationContext (c)
        {
          *this >> sources_ >> *this;
          *this >> schema_names_ >> ns_ >> names_ >> *this;
        }

        virtual Void
        traverse (SemanticGraph::Complex& c)
        {
          using SemanticGraph::Schema;

          traverse (static_cast<SemanticGraph::Type&> (c));

          if (c.inherits_p ())
          {
            SemanticGraph::Type& t (c.inherits ().base ());

            if (t.named () &&
                types_.find (
                  t.scope ().name () + L"#" + t.name ()) == types_.end ())
            {
              // Don't worry about types that are in included/imported
              // schemas.
              //
              Schema& s (dynamic_cast<Schema&> (t.scope ().scope ()));

              if (&s == &schema_root || sources_p (schema_root, s))
              {
                valid = false;

                wcerr << c.file () << ":" << c.line () << ":" << c.column ()
                      << ": error: type '" << xpath (c) << "' inherits from "
                      << "yet undefined type '" << xpath (t) << "'" << endl;

                wcerr << t.file () << ":" << t.line () << ":" << t.column ()
                      << ": info: '" << xpath (t) << "' is defined here"
                      << endl;

                wcerr << c.file () << ":" << c.line () << ":" << c.column ()
                      << ": info: inheritance from a yet-undefined type is "
                      << "not supported" << endl;

                wcerr << c.file () << ":" << c.line () << ":" << c.column ()
                      << ": info: re-arrange your schema and try again"
                      << endl;
              }
            }
          }
        }

        // Return true if root sources s.
        //
        Boolean
        sources_p (SemanticGraph::Schema& root, SemanticGraph::Schema& s)
        {
          using SemanticGraph::Schema;
          using SemanticGraph::Sources;

          for (Schema::UsesIterator i (root.uses_begin ());
               i != root.uses_end (); ++i)
          {
            if (i->is_a<Sources> ())
            {
              if (&i->schema () == &s || sources_p (i->schema (), s))
                return true;
            }
          }

          return false;
        }

        virtual Void
        traverse (SemanticGraph::Type& t)
        {
          if (t.named ())
          {
            types_.insert (t.scope ().name () + L"#" + t.name ());
          }
        }

      private:
        Containers::Set<String> types_;

        Traversal::Sources sources_;

        Traversal::Names schema_names_;
        Traversal::Namespace ns_;

        Traversal::Names names_;
      };

      //
      //
      struct AnonymousMember: protected ValidationContext
      {
        AnonymousMember (ValidationContext& c, Boolean& error_issued)
            : ValidationContext (c), error_issued_ (error_issued)
        {
        }

        Boolean
        traverse_common (SemanticGraph::Member& m)
        {
          SemanticGraph::Type& t (m.type ());

          if (!t.named ()
              && !t.is_a<SemanticGraph::Fundamental::IdRef> ()
              && !t.is_a<SemanticGraph::Fundamental::IdRefs> ())
          {
            if (!error_issued_)
            {
              valid = false;
              error_issued_ = true;

              wcerr << t.file ()
                    << ": error: anonymous types detected"
                    << endl;

              wcerr << t.file ()
                    << ": info: "
                    << "anonymous types are not supported in this mapping"
                    << endl;

              wcerr << t.file ()
                    << ": info: consider explicitly naming these types or "
                    << "remove the --preserve-anonymous option to "
                    << "automatically name them"
                    << endl;

              if (!options.value<CLI::show_anonymous> ())
                wcerr << t.file ()
                      << ": info: use --show-anonymous option to see these "
                      << "types" << endl;
            }

            return true;
          }

          return false;
        }

      private:
        Boolean& error_issued_;
      };

      struct AnonymousElement: Traversal::Element,
                               AnonymousMember
      {
        AnonymousElement (ValidationContext& c, Boolean& error_issued)
            : AnonymousMember (c, error_issued)
        {
        }

        virtual Void
        traverse (SemanticGraph::Element& e)
        {
          if (traverse_common (e))
          {
            if (options.value<CLI::show_anonymous> ())
            {
              wcerr << e.file () << ":" << e.line () << ":" << e.column ()
                    << ": error: element '" << xpath (e) << "' "
                    << "is of anonymous type" << endl;
            }
          }
          else
            Traversal::Element::traverse (e);
        }
      };

      struct AnonymousAttribute: Traversal::Attribute,
                                 AnonymousMember
      {
        AnonymousAttribute (ValidationContext& c, Boolean& error_issued)
            : AnonymousMember (c, error_issued)
        {
        }

        virtual Void
        traverse (Type& a)
        {
          if (traverse_common (a))
          {
            if (options.value<CLI::show_anonymous> ())
            {
              wcerr << a.file () << ":" << a.line () << ":" << a.column ()
                    << ": error: attribute '" << xpath (a) << "' "
                    << "is of anonymous type" << endl;
            }
          }
          else
            Traversal::Attribute::traverse (a);
        }
      };

      struct AnonymousType : Traversal::Schema,
                             Traversal::Complex,
                             protected virtual ValidationContext
      {
        AnonymousType (ValidationContext& c)
            : ValidationContext (c),
              error_issued_ (false),
              element_ (c, error_issued_),
              attribute_ (c, error_issued_)
        {
          *this >> sources_ >> *this;
          *this >> schema_names_ >> ns_ >> names_ >> *this;

          *this >> contains_compositor_ >> compositor_;
          compositor_ >> contains_particle_;
          contains_particle_ >> compositor_;
          contains_particle_ >> element_;

          *this >> names_attribute_ >> attribute_;
        }

      private:
        Boolean error_issued_;

        Containers::Set<String> types_;

        Traversal::Sources sources_;

        Traversal::Names schema_names_;
        Traversal::Namespace ns_;
        Traversal::Names names_;

        Traversal::Compositor compositor_;
        AnonymousElement element_;
        Traversal::ContainsCompositor contains_compositor_;
        Traversal::ContainsParticle contains_particle_;

        AnonymousAttribute attribute_;
        Traversal::Names names_attribute_;
      };
    }

    Validator::
    Validator ()
    {
      // Dummy ctor, help with long symbols on HP-UX.
    }

    Boolean Validator::
    validate (CLI::Options const& options,
              SemanticGraph::Schema& schema,
              SemanticGraph::Path const& tu)
    {
      Boolean valid (true);
      ValidationContext ctx (schema, options, valid);

      // Test for anonymout types.
      //
      {
        AnonymousType traverser (ctx);
        traverser.dispatch (schema);
      }


      // Test the rest.
      //
      if (valid)
      {
        Traverser traverser (ctx);
        traverser.dispatch (schema);
      }

      return valid;
    }
  }
}

