/*******************************************************************
 * EAPTTLS Function implementations
 * 
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * File: eapttls.c
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: eapttls.c,v 1.40 2005/11/12 23:59:53 chessing Exp $
 * $Date: 2005/11/12 23:59:53 $
 * $Log: eapttls.c,v $
 * Revision 1.40  2005/11/12 23:59:53  chessing
 * Some typo/debug patches from Pekka Savola.
 *
 * Revision 1.39  2005/10/17 03:56:55  chessing
 * Updates to the libxsupconfig library.  It no longer relies on other source from the main tree, so it can be used safely in other code with problems.
 *
 * Revision 1.38  2005/10/14 02:26:18  shaftoe
 * - cleanup gcc 4 warnings
 * - (re)add support for a pid in the form of /var/run/xsupplicant.<iface>.pid
 *
 * -- Eric Evans <eevans@sym-link.com>
 *
 * Revision 1.37  2005/09/14 03:34:55  chessing
 * Small cosmetic changes.  Default association mode is now auto instead of manual. Fixes for bug IDs #1290449 & #1290323.
 *
 * Revision 1.36  2005/08/25 03:34:06  chessing
 * Removed a bunch of functions from config.c that could be handled better in other ways.
 *
 * Revision 1.35  2005/08/25 02:20:20  chessing
 * Some cleanup in xsup_debug.c, added the ability to wait for an interface to come up if it is down when Xsupplicant is started.  Roughed in the ability to choose between having Xsupplicant auto associate you, or allow you to set the ssid manually.  The stale key timer can now be set in the config file.  The association timeout can be set in the config file, and will also be used once the functionality is in place to attempt to guess the key settings needed for association, as well as the functionality to auto associate.
 *
 * Revision 1.34  2005/08/09 01:39:18  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/

#include <string.h>
#include <stdlib.h>

#include "profile.h"
#include "xsupconfig.h"
#include "xsup_debug.h"
#include "xsup_err.h"
#include "frame_structs.h"
#include "eap_types/tls/eaptls.h"
#include "eap_types/ttls/eapttls.h"
#include "eap_types/ttls/ttlsphase2.h"
#include "eap_types/tls/tls_funcs.h"
#include "eap.h"

int eapttls_setup(struct generic_eap_data *thisint)
{
  struct tls_vars *mytls_vars;
  struct config_eap_ttls *userdata;
  struct config_globals *globals;

  if (!thisint)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface structure passed to eapttls_setup()!\n");
      return XEMALLOC;
    }

  userdata = (struct config_eap_ttls *)thisint->eap_conf_data;

  if (userdata == NULL)
    {
      debug_printf(DEBUG_NORMAL, "EAP-TTLS setup was passed NULL userdata!  We cannot continue with this attempt at authentication!\n");
      return XENOUSERDATA;
    }

  // First, set up the structure to hold all of our instance specific
  // variables.
  thisint->eap_data = (char *)malloc(sizeof(struct tls_vars));
  if (thisint->eap_data == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Unable to allocate memory for thisint->eapdata in eapttls_setup()!\n");
      return XEMALLOC;
    }

  mytls_vars = (struct tls_vars *)thisint->eap_data;
  bzero(mytls_vars, sizeof(struct tls_vars));

  // Set our variables to NULL.
  mytls_vars->ctx = NULL;
  mytls_vars->ssl = NULL;
  mytls_vars->ssl_in = NULL;
  mytls_vars->ssl_out = NULL;
  mytls_vars->tlsoutdata = NULL;
  mytls_vars->tlsoutsize = 0;
  mytls_vars->tlsoutptr = 0;
  mytls_vars->cncheck = userdata->cncheck;
  mytls_vars->cnexact = userdata->cnexact;
  mytls_vars->phase = 1;
  mytls_vars->resume = userdata->session_resume;
  mytls_vars->quickResponse = TRUE;
  mytls_vars->cert_loaded = FALSE;
  mytls_vars->verify_mode = SSL_VERIFY_PEER;

  mytls_vars->sessionkeyconst = (char *)malloc(TTLS_SESSION_KEY_CONST_SIZE+1);
  if (mytls_vars->sessionkeyconst == NULL) return XEMALLOC;

  bzero(mytls_vars->sessionkeyconst, TTLS_SESSION_KEY_CONST_SIZE+1);
  strncpy(mytls_vars->sessionkeyconst, TTLS_SESSION_KEY_CONST,
	  TTLS_SESSION_KEY_CONST_SIZE);

  mytls_vars->sessionkeylen = TTLS_SESSION_KEY_CONST_SIZE;

  debug_printf(DEBUG_EVERYTHING, "(EAP-TTLS) Initialized.\n");
  
  if (tls_funcs_init(thisint) != XENONE) 
    {
      debug_printf(DEBUG_NORMAL, "Couldn't initialize OpenSSL!\n");
      return XETLSINIT;
    }

  if ((!userdata->root_cert) && (!userdata->root_dir))
    {
      debug_printf(DEBUG_NORMAL, "userdata->root_cert is NULL!  We cannot continue without a valid certificate!\n");
      return XENOUSERDATA;
    }

  if ((userdata->root_cert) && (strcmp(userdata->root_cert, "NONE") == 0))
    {
      // We were told not to verify certificates.  Spew out a warning, and
      // then do it!
      mytls_vars->verify_mode = SSL_VERIFY_NONE;

      globals = config_get_globals();

      if (globals != NULL)
	{
	  if (!TEST_FLAG(globals->flags, CONFIG_GLOBALS_NO_FRIENDLY_WARNINGS))
	    {
	      debug_printf(DEBUG_NORMAL, "****WARNING**** Turning off certificate "
			   "verification is a *VERY* bad idea!  You should not "
			   "use this mode outside of basic testing, as it will "
			   "compromise the security of your connection!\n");
	    }
	}
    } else {
      if (tls_funcs_load_root_certs(thisint, userdata->root_cert, 
				    userdata->root_dir, userdata->crl_dir) != XENONE)
	{
	  debug_printf(DEBUG_NORMAL, "Couldn't load root certificates!\n");
	  return XETLSINIT;
	}
    }

  if ((userdata->user_cert != NULL) && ((userdata->user_key_pass != NULL) ||
				   (thisint->tempPwd != NULL)))
    {
      debug_printf(DEBUG_NORMAL, "Using user certificate with TTLS!\n");
      tls_funcs_load_user_cert(thisint, userdata->user_cert, 
			       userdata->user_key, userdata->user_key_pass,
			       userdata->random_file);

      mytls_vars->cert_loaded = TRUE;
    }

  // Otherwise, if we don't have a user cert, then just set cert_loaded to true!
  if (userdata->user_cert == NULL) mytls_vars->cert_loaded = TRUE;

  return XENONE;
}

int eapttls_process(struct generic_eap_data *thisint, u_char *dataoffs, 
		    int insize, u_char *outframe, int *outsize)
{
  struct config_eap_ttls *userdata;
  struct tls_vars *mytls_vars;
  int result = XENONE;
  int retVal;

  if ((!thisint) || (!thisint->eap_conf_data) || (!thisint->eap_data) ||
      (!outframe))
    {
      debug_printf(DEBUG_NORMAL, "Invalid data passed to eapttls_process()!\n");
      return XEMALLOC;
    }

  userdata = (struct config_eap_ttls *)thisint->eap_conf_data;
  mytls_vars = (struct tls_vars *)thisint->eap_data;

  // The state machine wants to know if we have anything else to say.
  // We may be waiting for the server to send us more information, or
  // we may need to send a request to the GUI for a password, and wait
  // for an answer.

  // TTLS is slightly different than others.  Since we don't *need* to have
  // a client certificate to make things work correctly, we may not need
  // a password here.
  if (mytls_vars->cert_loaded == FALSE)
    {
      if (userdata->user_cert != NULL)
	{
	  if ((thisint->tempPwd == NULL) && (userdata->user_key_pass == NULL))
	    {
	      // We need to indicate to the caller that we need a password.
	      thisint->need_password = 1;
	      thisint->eaptype = strdup("EAP-TTLS User Certificate");
	      thisint->eapchallenge = NULL;   // Probably not needed, but just
                                              // to be safe.

	      *outsize = 0;

	      return XENONE;
	    }
	}

      if ((mytls_vars->cert_loaded == FALSE) && ((thisint->tempPwd != NULL) ||
						 (userdata->user_key_pass != NULL)))
      {
	// Load the user certificate.
	if ((retVal = tls_funcs_load_user_cert(thisint, userdata->user_cert, 
					       userdata->user_key,
					       userdata->user_key_pass,
					       userdata->random_file))!=XENONE)
	  {
	    debug_printf(DEBUG_NORMAL, "Error loading user certificate!\n");
	    return retVal;
	  } else {

	    // Otherwise, the certificate is loaded.
	    mytls_vars->cert_loaded = TRUE;

	    // Check if we used tempPwd, if we did, free it, since we don't
	    // need it anymore.
	    if (thisint->tempPwd != NULL)
	      {
		free(thisint->tempPwd);
		thisint->tempPwd = NULL;
	      }
	  }
      }  
    }


  // Make sure we have a valid packet to process.
  if (dataoffs == NULL) return XENONE;
  
  result=tls_funcs_decode_packet(thisint, (char *) dataoffs, insize, (char *) outframe, outsize,
				 (phase2_call)ttls_do_phase2, userdata->chunk_size);

  if (result != 0)
    {
      return result;
    }
  return XENONE; //*outsize;
}

int eapttls_get_keys(struct interface_data *thisint)
{
  struct config_network *network_data;

  if (!thisint)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface structure passed to %s()!"
		   "\n", __FUNCTION__);
      return XEMALLOC;
    }

  network_data = config_get_network_config();
  
  if (network_data == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid network configuration structure! ("
		   "%s:%d)\n", __FUNCTION__, __LINE__);
      return XEBADCONFIG;
    }

  if (thisint->keyingMaterial != NULL)
    {
      free(thisint->keyingMaterial);
    }
  
  thisint->keyingMaterial = (u_char *) tls_funcs_gen_keyblock(network_data->activemethod);
  thisint->keyingLength = 32;

  if (thisint->keyingMaterial == NULL) return -1;

  return 0;
}

int eapttls_cleanup(struct generic_eap_data *thisint)
{
  struct tls_vars *mytls_vars;

  if ((!thisint) || (!thisint->eap_data))
    {
      debug_printf(DEBUG_NORMAL, "Invalid structure in eapttls_cleanup()!\n");
      return XEMALLOC;
    }

  mytls_vars = (struct tls_vars *)thisint->eap_data;
  tls_funcs_cleanup(thisint);

  if (mytls_vars != NULL)
    {
      free(mytls_vars);
      mytls_vars = NULL;
    }

  debug_printf(DEBUG_EVERYTHING, "(EAP-TTLS) Cleaned up.\n");
  return XENONE;
}

int eapttls_failed(struct generic_eap_data *thisint)
{
  struct tls_vars *mytls_vars;

  if ((!thisint) || (!thisint->eap_data))
    {
      debug_printf(DEBUG_NORMAL, "Invalid structure passed to eapttls_failed()!\n");
      return XEMALLOC;
    }

  // Call ttls_phase2_failed first.
  ttls_phase2_failed(thisint);

  mytls_vars = (struct tls_vars *)thisint->eap_data;
  tls_funcs_failed(thisint);

  debug_printf(DEBUG_EVERYTHING, "(EAP-TTLS) Failed. Resetting.\n");
  return XENONE;
}
