/*
 * YH - Console Chinese Environment -
 * Copyright (C) 1999 Red Flag Linux (office@sonata.iscas.ac.cn)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY TAKASHI MANABE ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE TERRENCE R. LAMBERT BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */


/*****************---ansi.c----*********************/


#include <stdio.h>
#if 0	
#include <sys/ascii.h>
#endif
#include "ascii.h"

#ifdef OPENSERVER
#include <sys/keyboard.h>
#else
#include <sys/types.h>
#include <sys/kd.h>

#define A_SP		32
#define K_FUNL		122			/* Last function key */
#define K_FUNF		27			/* First function key */
#define NFKEYS		(K_FUNL - K_FUNF + 1)
#define FKEYSTART	0			/* The F1 key is string key #0 */
#endif

#include <sys/types.h>
#include "screen.h"
#include "display.h"

/*
 * Values for the states of the
 * ANSI parser state machine.
 */
#define CSS_START	0
#define CSS_ESC		1
#define CSS_CSI		2
#define CSS_SCO		3
#define CSS_QUES1	4
#define CSS_QUES2	5
#define CSS_Q		20
#define CSS_QNUM	21
#define CSS_QNUMDELM	22
#define CSS_QSTRCAP	23
#define CSS_HASH	24	

struct screen	g_scrn;
extern int fdm;	


/* map ANSI color numbers to SCO color numbers */
char ansicolormap[8] =
{
	BLACK, RED, GREEN, BROWN,
	BLUE, MAGENTA, CYAN, WHITE
};

int bottom_disable = 0;
int	mono = 0;


/*******************************functions***********************************/

void ansi(cp, nchars)
u_char *cp;			/* front 'ptr' to characters to output	*/
int nchars;			/* number of characters to output	*/
{
	/* pointer to the screen to output to   */
	register struct screen *msp = &g_scrn;

	register unsigned c;	/* current character being parsed	*/
	u_char *bp;		/* 'back' ptr for blocks of characters	*/
	int i;

	void (*pchar)()=msp->mv_adapter->v_pchar;

	if(!TEXTMODE(msp))
		return;
	/*
	 * Use msp->mv_tmpcol rather than msp->mv_col so we can batch
	 * up lines of characters into one call into the
	 * adapter driver.
	 */
	if (msp->mv_curflg & CUR_ON)
		vga1_scurs(ECHO_OFF);

	msp->mv_tmpcol = msp->mv_col;
	bp = cp;

	while (--nchars >= 0)
	{
		c = *cp;

		if (msp->mv_csstate == CSS_START)
		{
			switch (msp->mv_font)
			{
			case 0:
				if (c < A_SP)
					goto nodisp;
				break;
			case 1:
				if (c == A_ESC)
					goto nodisp;
				break;
			case 2:
				if (c == A_ESC)
					goto nodisp;
#ifdef OPENSERVER
				*cp ^= 0x80; 
#endif
				break;
			}

			/*
			 * The character is a normal display (graphic) character.
			 * so advance the front pointer.
			 */
			++cp;

			if(msp->mv_wrap)  
			{
				if(msp->mv_row==msp->mv_rsz-1)	
					(*msp->mv_adapter->v_scroll)(msp,1);
				else
					msp->mv_row++;
				msp->mv_col=msp->mv_tmpcol=0;
			}
			
			/*
			 * If we reached the end of the line then display what
			 * we have before going to the next line.
			 */
			if (++msp->mv_tmpcol >= msp->mv_csz)
			{
				(*pchar)(msp, bp, cp-bp);

				bp = cp;
				msp->mv_wrap=1;  
#if 0	 
				msp->mv_col = msp->mv_tmpcol;
#endif
			}
			else	
				msp->mv_wrap=0;
			continue;	/* for normal characters loop back to the top */

			/*
			 * If we have a non-printing character then display any
			 * saved up printing characters before processing the
			 * special character.   After a saved up block has been
			 * emmited, sync up mv_col and bp.
			 */
nodisp:
			if (i = cp - bp)
				(*pchar)(msp, bp, i);
#if 1 
			msp->mv_col = msp->mv_tmpcol;
#endif
			bp = ++cp;

			switch(c)
			{
			case A_ESC:
				msp->mv_csstate = CSS_ESC;
				break;

newline:
				msp->mv_col = msp->mv_tmpcol = 0;
				/* fall through */
			case A_LF:
				if (msp->mv_row == msp->mv_rsz - 1) 
				{
					(*msp->mv_adapter->v_scroll) (msp, 1);
				}
				else
					msp->mv_row++;
				msp->mv_wrap=0;	
				break;

			case A_CR:
				msp->mv_col = msp->mv_tmpcol = 0;
				msp->mv_wrap=0; 
				break;

			case A_BS:
				if (msp->mv_curflg & CUR_ON)
					vga1_scurs(ECHO_OFF);
				if (msp->mv_tmpcol > 0)
					msp->mv_col = --msp->mv_tmpcol;
				msp->mv_wrap=0; 
				break;

			case A_HT:
				i = 8 - (msp->mv_tmpcol & 07);
				if ((msp->mv_tmpcol += i) >= msp->mv_csz)
					goto newline;
				msp->mv_col = msp->mv_tmpcol;
				break;

			case A_BEL:
				bell(msp->mb_freq, msp->mb_time);
				break;

			case A_FF:				/* NON-ANSI */
				(*msp->mv_adapter->v_clear) (msp, msp->mv_top, 0, msp->mv_csz*(msp->mv_rsz-msp->mv_top));
				msp->mv_row = msp->mv_top;
				msp->mv_col = msp->mv_tmpcol = 0;
				break;

			case A_NUL:
				break;

		/*	case A_CSI:		* 8 bit "ESC [" function *
				goto got_csi;  */

			default:
				break;		/* non-special cntl chars.  (eg. ^A )	*/
			}
			continue;		/* back to top to process next character */
		}

		/*
		 * We are not in the start state.  Do not let bp
		 * fall behind cp.
		 */
		bp = ++cp;

		switch (msp->mv_csstate)
		{
		default:
			break;
			/*
			 * Process characters that come immed. after an ESC.
			 */
		case CSS_ESC:
			switch(c)
			{
			case '[':
got_csi:
				msp->mv_csstate = CSS_CSI;
				msp->mv_csindex = 0;
				for (i=0; i<NCSPARAMS; i++)
					msp->mv_csparam[i] = 0;
				break;

			case 'Q':		/* non-ANSI set function key		*/
				msp->mv_csstate = CSS_Q;
				msp->mv_csindex = 0;
				for (i=0; i<NCSPARAMS; i++)
					msp->mv_csparam[i] = 0;
				break;

			case '7':		/* non-ANSI save cursor postion		*/
				msp->mf_savrow = msp->mv_row;
				msp->mf_savcol = msp->mv_col;
				msp->mv_csstate = CSS_START;
				break;

			case '8':		/* non-ANSI restore cursor postion	*/
				msp->mv_row = msp->mf_savrow;
				msp->mv_tmpcol = msp->mv_col = msp->mf_savcol;
				msp->mv_csstate = CSS_START;
				msp->mv_wrap=0; 
				break;

			case 'E':		
				msp->mv_col=msp->mv_tmpcol=0;	
				if(msp->mv_row==msp->mv_rsz-1)
					(*msp->mv_adapter->v_scroll)(msp,1);
				else
					msp->mv_row++;
				msp->mv_csstate=CSS_START;
				break;
			
			case 'M':		
				if (msp->mv_row==msp->mv_top)	
					(*msp->mv_adapter->v_scroll)(msp,-1);
				else if (msp->mv_row>msp->mv_top)
					msp->mv_row--;
				msp->mv_csstate=CSS_START;
				msp->mv_wrap=0;
				break;

			case 'D':		
				if(msp->mv_row==msp->mv_rsz-1)
					(*msp->mv_adapter->v_scroll)(msp,1);
				else
					msp->mv_row++;
				msp->mv_csstate=CSS_START;
				break;

			case 'H':		
			msp->mv_tabstop[msp->mv_col>>5]|=(1<<(msp->mv_col&31));
			msp->mv_csstate=CSS_START;
			break;

			case 'Z':		
				write(fdm,"\033[?6c",strlen("\033[?6c"));
				msp->mv_csstate=CSS_START;
				break;

			case 'c':		
				initdisplay();
				msp->mv_csstate=CSS_START;
				break;

			case '>':
				write(0,"\033>",2);
				msp->mv_csstate=CSS_START;
				break;

			case '=':		
				write(0,"\033=",2);
				msp->mv_csstate=CSS_START;
				break;

			case '#':	
				msp->mv_csstate=CSS_HASH;
				break;
				
			default:
				msp->mv_csstate = CSS_START;
				break;
			}
			break;		/* end of CSS_ESC */
			/*
			 * Function key programming.
			 */
		case CSS_Q:		/* ESC Q */
		case CSS_QNUM:		/* ESC Q <num> */
		case CSS_QNUMDELM:	/* ESC Q <num><delm>	read string here */
		case CSS_QSTRCAP:	/* contrl characters escaped with '^' */
			ansi_qstr(msp, c);
			break;
		
		case CSS_HASH:		
			msp->mv_csstate=CSS_START;
			if (c=='8')
			{
			int i,j,oldfont;
			oldfont=msp->mv_font;
			msp->mv_font=0;
			for(i=msp->mv_top;i<msp->mv_rsz;i++)
			  for(j=0;j<msp->mv_csz;j++)
			    {msp->mv_col=j;
			     msp->mv_row=i;
			     (*pchar)(msp,'E',1);}
			 msp->mv_font=oldfont;
			 msp->mv_csstate=CSS_START;
			}
			break; 

			/*
			 * ESC [ ........
			 */
		case CSS_CSI:
			switch (c)
			{
			case '0':
			case '1':
			case '2':
			case '3':
			case '4':
			case '5':
			case '6':
			case '7':
			case '8':
			case '9':
				i = msp->mv_csindex;
				msp->mv_csparam[i] *= 10;
				msp->mv_csparam[i] += c - '0';
				break;

			case ';':
				if (++msp->mv_csindex >= NCSPARAMS)
					msp->mv_csstate = CSS_START;
				break;

			case '=':		/* Private SCO sequences */
				msp->mv_csstate = CSS_SCO;
				break;

			case '?':
				/*msp->mv_csstate = CSS_QUES1;*/
				break;
				/*
				 * Cursor motions.
				 */
			case 'A':		/* CUU (CUrsor Up) */
			case 'B':		/* CUD (CUrsor Down) */
			case 'C':		/* CUF (CUrsor Forward) */
			case 'D':		/* CUB (CUrsor Back) */
			case 'Z':		/* CBT (Cursor Backward Tabulation) */

			case 'F':		/* CPL (Cursor to Previous Line) */
			case 'E':		/* CNL (Cursor to Next Line) */
			case 'e':		/* VPR (Vertical Position Relative) */
			case 'd':		/* VPA (Vertical Position Absolute) */
			case 'a':		/* HPR (Horizontal Position Relative) */
			case '`':		/* HPA (Horizontal Position Absolute) */
			case 'G':		

			case 'H':		/* CUP (CUrsor Position) */
			case 'f':		/* HVP (Horizontal and Vertical Position) */
				ansi_csrmove(msp, c);
				msp->mv_csstate = CSS_START;
				msp->mv_wrap=0;  
				break;

				/*
				 * Erase things.
				 */
			case 'J':		/* ED  (Erase in Display) */
			case 'K':		/* EL  (Erase in Line) */
			case 'X':		/* ECH (Erase CHaracter) */
				ansi_erase(msp, c);
				msp->mv_csstate = CSS_START;
				msp->mv_wrap=0; 
				break;

				/*
				 * Print exact font characters.
				 */
			case 'g':		/* should be but isn't ANSI Tabulation Clear */
				ansi_vio(msp, c, 0);
				msp->mv_csstate = CSS_START;
				if(msp->mv_tmpcol >= msp->mv_csz)
					goto newline;
				break;

			case 's':		/* MS-DOS save cursor postion	*/
				msp->mf_savrow = msp->mv_row;
				msp->mf_savcol = msp->mv_col;
				msp->mv_csstate = CSS_START;
				break;

			case 'u':		/* MS-DOS restore cursor postion */
				msp->mv_row = msp->mf_savrow;
				msp->mv_tmpcol = msp->mv_col = msp->mf_savcol;
				msp->mv_csstate = CSS_START;
				msp->mv_wrap=0;  
				break;

			case 'i':		/* Send screen to host */

				break;
				/*
				 * Lock and unlock the keyboard.
			 	 * you can now hang the console just like any smart terminal!!
				 */
			case 'h':		/* lock the keyboard */
				if(msp->mv_curflg&CUR_HIDDEN) vga1_scurs(ECHO_ON);
				msp->mv_csstate = CSS_START;
				break;

			case 'r':    
				msp->mv_top=msp->mv_csparam[0]-1;
				msp->mv_rsz=msp->mv_csparam[1];
				msp->mv_savsz=msp->mv_rsz*msp->mv_csz;
				msp->mv_csstate=CSS_START;
				msp->mv_wrap=0;   
				break;	
			
			case 'c':	
				if(!msp->mv_csparam[0])
				       write(fdm,"\033[?6c",strlen("\033[?6c"));
				msp->mv_csstate=CSS_START;
				break;

			case 'l':		/* unlock the keyboard */
				if(msp->mv_curflg&CUR_ON) vga1_scurs(ECHO_OFF);
				msp->mv_csstate = CSS_START;
				break;

			case 'z':		/* switch screen */
				msp->mv_csstate=CSS_START;
				break;

			case 'm':		/* SGR (Set Graphic Rendition) */
				ansi_sgr(msp);
				msp->mv_csstate = CSS_START;
				break;
				/*
				 * Move data around on the screen.
				 */
			case '@':		/* ICH (Insert CHaracter) */
			case 'P':		/* DCH (Delete CHaracter) */
			case 'L':		/* IL (Insert Line) */
			case 'M':		/* DL (Delete Line) */
			case 'S':		/* SU (Scroll Up) */
			case 'T':		/* SD (Scroll Down) */
				ansi_datamove(msp, c);
				msp->mv_csstate = CSS_START;
				break;

				/* add escape sequence for output */

			case ']':           /* for write bottom noted line */

				if(bottom_disable==1)
				{
					msp->mv_csstate=CSS_START;
					return;
				}

/*				PutCC16(msp,0,25,6,(char *)cp); */
				PutCC16(msp,0,25,7,(char *)cp);
				msp->mv_csstate=CSS_START;

				return ;

			case '{':		/* for make the character black */

				bottom_disable=1;
				ClearBottom(msp);
				msp->mv_csstate=CSS_START;

				return;

			case '}':          /* restore grey */

				bottom_disable=0;
				Writebackground(msp);
				redrawbottom(msp,0,25,7);
				msp->mv_csstate=CSS_START;

				return;

				/* add escape sequence end */

			default:
				msp->mv_csstate = CSS_START;
				break;

			}
			break;			/* end of CSS_CSI switch */

		case CSS_SCO:
			ansi_sco(msp, c);
			break;

		case CSS_QUES1:
		case CSS_QUES2:
			ansi_question(msp, c);
			break;
		}
	} /* end while */

	/*
	 * flush out any remaining saved characters at the
	 * end.
	 */
	if (i = cp - bp)
	{
		(*pchar)(msp, bp, i);
	}
#if 1
	msp->mv_col = msp->mv_tmpcol;
#endif
	/*
	    (*(msp->mv_adapter->v_scurs))(msp);
	*/

}


ansi_qstr(msp, c)
struct screen *msp;
int c;
{
	switch(msp->mv_csstate)
	{
	case CSS_Q:		/* ESC Q */
		c = c - '0';
		if(c < 0 || c >= NFKEYS)
			msp->mv_csstate = CSS_START;
		else
		{
			msp->mv_csparam[0] = c + FKEYSTART;
			msp->mv_csstate = CSS_QNUM;
		}
		break;

	case CSS_QNUM:				/* ESC Q <num> */
		msp->mv_csparam[1] = c;		/* string delimiter */
		msp->mv_csparam[2] = 0;		/* string length */
		msp->mv_csstate = CSS_QNUMDELM;
		break;

	case CSS_QNUMDELM:	 /* ESC Q <num><delm>	read string here */
		if( c == msp->mv_csparam[1] )
		{
			msp->mv_csstate = CSS_START;
		}
		else if( c == '^')
			msp->mv_csstate = CSS_QSTRCAP;
		else if( c != '\0')			/* NOTE: can't hack nulls */
		{
			msp->mk_qstr[msp->mv_csparam[2]++] = c;
			if( msp->mv_csparam[2] >= MAXFK)	/* buffer full ? */
				msp->mv_csstate = CSS_START;
		}
		break;

	case CSS_QSTRCAP:	/* contrl characters escaped with '^' */

		c -= ' ';
		if( c != '\0')	/* NOTE: can't hack nulls */
		{
			msp->mk_qstr[msp->mv_csparam[2]++] = c;
			if( msp->mv_csparam[2] >= MAXFK)	/* buffer full ? */
				msp->mv_csstate = CSS_START;
		}
		msp->mv_csstate = CSS_QNUMDELM;		/* S001 */
		break;
	}
}


ansi_csrmove(msp, c)
struct screen *msp;
int c;
{
	int tab;
	switch(c)
	{
	case 'H':		/* CUP (CUrsor Position) */
	case 'f':		/* HVP (Horizontal and Vertical Position) */
		msp->mv_row = ansirange(msp->mv_csparam[0], msp->mv_rsz) - 1;
		msp->mv_col = ansirange(msp->mv_csparam[1], msp->mv_csz) - 1;
		msp->mv_tmpcol = msp->mv_col;
		break;

	case 'd':		/* VPA (Vertical Position Absolute) */
		msp->mv_row = ansirange(msp->mv_csparam[0], msp->mv_rsz) - 1;
		break;

	case 'F':		/* CPL (Cursor to Previous Line) */
		msp->mv_tmpcol = msp->mv_col = 0;
		/* fall through */

	case 'A':		/* CUU (CUrsor Up) */
		msp->mv_row -= ansirange(msp->mv_csparam[0], msp->mv_row);
		break;

	case 'E':		/* CNL (Cursor to Next Line) */
		msp->mv_tmpcol = msp->mv_col = 0;

	case 'B':		/* CUD (CUrsor Down) */
	case 'e':		/* VPR (Vertical Position Relative) */
		msp->mv_row += ansirange(msp->mv_csparam[0],
		    (msp->mv_rsz -1) - msp->mv_row);

		break;

	case 'C':		/* CUF (CUrsor Forward) */
	case 'a':		/* HPR (Horizontal Position Relative) */
		msp->mv_tmpcol = msp->mv_col += ansirange(msp->mv_csparam[0],
		    (msp->mv_csz -1) - msp->mv_col);
		break;

	case 'D':		/* CUB (CUrsor Back) */
		msp->mv_col -= ansirange(msp->mv_csparam[0], msp->mv_col);
		msp->mv_tmpcol = msp->mv_col;
		break;

	case 'Z':		/* CBT (Cursor Backward Tabulation) */
		tab = ansirange(msp->mv_csparam[0], (msp->mv_col + 7) >> 3);
		if (msp->mv_col & 7)
		{
			msp->mv_col &= ~7;
			if(tab)
				--tab;
		}
		msp->mv_tmpcol = msp->mv_col -= tab << 3;
		break;

	case '`':		/* HPA (Horizontal Position Absolute) */
	case 'G':
		msp->mv_col = ansirange(msp->mv_csparam[0], msp->mv_csz) - 1;
		msp->mv_tmpcol = msp->mv_col;
		break;
	}
	/*    (*(msp->mv_adapter->v_scurs))(msp);*/

}


ansi_datamove(msp, c)
struct screen *msp;
int c;
{
	void (*copy)()=msp->mv_adapter->v_copy;
	void (*clear)()=msp->mv_adapter->v_clear;
	void (*scroll)()=msp->mv_adapter->v_scroll;

	int tail, amount;

	switch(c)
	{
	case '@':		/* ICH (Insert CHaracter) */
		tail = msp->mv_csz - msp->mv_col;
		amount = ansirange(msp->mv_csparam[0], tail);
		tail -= amount;
		if(tail)
			(*copy)(msp, msp->mv_row, msp->mv_col+amount,
			    msp->mv_row, msp->mv_col, tail);
		(*clear)(msp, msp->mv_row, msp->mv_col, amount);
		break;

	case 'P':		/* DCH (Delete CHaracter) */
		tail = msp->mv_csz - msp->mv_col;
		amount = ansirange(msp->mv_csparam[0], tail);
		tail -= amount;
		if(tail)
			(*copy)(msp, msp->mv_row, msp->mv_col,
			    msp->mv_row, msp->mv_col+amount, tail);
		(*clear)(msp, msp->mv_row, msp->mv_col + tail, amount);
		break;

	case 'L':	/* IL (Insert Line) */
		tail = msp->mv_rsz - msp->mv_row;
		amount = ansirange(msp->mv_csparam[0], tail);
		tail -= amount;
		if(tail && 0==msp->mv_row)
		{
			(*scroll)(msp, -amount);	/* optimization	 */
			break;
		}
		if(tail)
			(*copy)(msp, amount+msp->mv_row, 0,
			    msp->mv_row, 0, tail*msp->mv_csz);
		(*clear)(msp, msp->mv_row, 0, amount*msp->mv_csz);
		break;

	case 'M':		/* DL (Delete Line) */
		tail = msp->mv_rsz - msp->mv_row;
		amount = ansirange(msp->mv_csparam[0], tail);
		tail -= amount;
		if(tail)
			(*copy)(msp, msp->mv_row, 0,
			    amount+msp->mv_row, 0, tail*msp->mv_csz);
		(*clear)(msp, tail+msp->mv_row, 0, amount*msp->mv_csz);
		break;

	case 'S':		/* SU (Scroll Up) */
		amount = ansirange(msp->mv_csparam[0], msp->mv_rsz);
		if(amount == msp->mv_rsz)
			(*clear)(msp, 0, 0, SCREENSZ(msp));
		else
			(*scroll)(msp, amount);
		break;

	case 'T':		/* SD (Scroll Down) */
		amount = ansirange(msp->mv_csparam[0], msp->mv_rsz);
		if(amount == msp->mv_rsz)
			(*clear)(msp, 0, 0, SCREENSZ(msp));
		else
			(*scroll)(msp, -amount);
		break;
	}
}


ansi_erase(msp, c)
struct screen *msp;
int c;
{
	int soff, ssize, amount;
	void (*clear)()=msp->mv_adapter->v_clear;

	switch(c)
	{
	case 'J':		/* ED (Erase in Display) */
		soff = CURSOR(msp);
		ssize = SCREENSZ(msp);
		switch(msp->mv_csparam[0])
		{
		case 0:			/* current pos to end of screen	*/
			(*clear) (msp, msp->mv_row, msp->mv_col, ssize-soff);
			break;

		case 1:			/* top of screen to current pos	*/
			(*clear) (msp, 0, 0, 1 + soff);
			break;

		case 2:			/* clear whole screen */
			(*clear) (msp, 0, 0, ssize);
			msp->mv_row = msp->mv_col = 0;	/* NON-ANSI VT100ism */
			break;
		}
		break;

	case 'K':		/* EL (Erase in Line) */
		switch(msp->mv_csparam[0])
		{
		case 0:			/* pos to end of line */
			(*clear)(msp, msp->mv_row, msp->mv_col, msp->mv_csz - msp->mv_col);
			break;

		case 1:			/* start of line to pos */
			(*clear)(msp, msp->mv_row, 0, 1+msp->mv_col);
			break;

		case 2:			/* whole line */
			(*clear)(msp, msp->mv_row, 0, msp->mv_csz);
			break;
		}
		break;

	case 'X':		/* ECH (Erase CHaracter) */
		amount = ansirange(msp->mv_csparam[0], msp->mv_csz - msp->mv_col);
		(*clear)(msp, msp->mv_row, msp->mv_col, amount);
		break;
	}
}


ansi_sgr(msp)
struct screen *msp;
{
	int i, color;
	void (*sgr)()=msp->mv_adapter->v_sgr;

	switch (msp->mv_csparam[0])
	{
	case SGR_PRCOLORS:
	case SGR_PRBLKCTL:
	case SGR_REVERSE:
		ansi_vio(msp, 'm', 0);
		return;
	}

	i = 0;
	while (i <= msp->mv_csindex)
	{
		switch (msp->mv_csparam[i])
		{
		case SGR_FONT+0:
		case SGR_FONT+1:
		case SGR_FONT+2:
			ansi_vio(msp, 'm', i);
			break;

		default:
			(*sgr)(msp, msp->mv_csparam[i]);
			break;
		}
		++i;
	}
}


ansi_vio(msp, c, i)
struct screen *msp;
int c, i;
{
	void (*sgr)()=msp->mv_adapter->v_sgr;
	int (*adapctl)()=msp->mv_adapter->v_adapctl;

	switch(c)
	{
	case 'm':		/* Select Graphic Rendition */
		switch (msp->mv_csparam[i])
		{
			/*
			 * Set Fore and Back Colors and change current colors
			 * to normal colors.
			 */
		case SGR_PRCOLORS:
			(*adapctl)(msp, AC_DEFNF, msp->mv_csparam[1]);
			(*adapctl)(msp, AC_DEFNB, msp->mv_csparam[2]);
			(*adapctl)(msp, AC_ONN, 0);
			break;

			/*
			 * Set and clear the blink bit of an IBM card.
			 * ANSI defines ESC[3m to be "italic"
			 */
		case SGR_PRBLKCTL:		/* Set/Clear Blink bit */
			(*adapctl)(msp, AC_BLINKB, msp->mv_csparam[1]);
			break;

			/*
			 * ANSI defines ESC[7m to be "reverse"
			 * This agrees with that but, it also does:
			 *   ESC[7;xx;yym  which sets the reverse colors
			 * to be xx, yy.
			 */
		case SGR_REVERSE:		/* Reverse video */
			if(msp->mv_csindex == 0)
				(*sgr)(msp, 7);		/* start reverse */
			else
			{
				(*adapctl)(msp, AC_DEFRF, msp->mv_csparam[1]);
				(*adapctl)(msp, AC_DEFRB, msp->mv_csparam[2]);

			}
			msp->mf_status &= ~MFS_TMODES;
			msp->mf_status |=  MFS_REVSTEXT;
			break;

		case SGR_FONT+0:
			msp->mv_font = 0;
			msp->mf_status &= ~MFS_TMODES;
			break;
		case SGR_FONT+1:
		case SGR_FONT+2:
			msp->mv_font = 2;
			msp->mf_status &= ~MFS_TMODES;
			msp->mf_status |= MFS_GRAFTEXT;

			break;
		}
		break;	/* end of case 'm' */

	case 'g':			/* Output font character by number */
		if(msp->mv_csindex == 0)	/* in the "graphic colors" */
		{
			(*adapctl)(msp, AC_ONG, 0);
			(*adapctl)(msp, AC_FONTCHAR, msp->mv_csparam[0]);
			(*adapctl)(msp, AC_ONN, 0);

			msp->mv_csstate = CSS_START;
			msp->mv_col = ++msp->mv_tmpcol;
			return;
		}
		(*adapctl)(msp, AC_DEFGF, msp->mv_csparam[1]);
		(*adapctl)(msp, AC_DEFGB, msp->mv_csparam[2]);
		break;
	}
}


/*
 * SCO question mark sequences.  In the "ESC [ ?" range
 */
ansi_question(msp, c)
struct screen *msp;
int c;
{
	switch(msp->mv_csstate)
	{
	case CSS_QUES1:
		if('7' == c)
			msp->mv_csstate = CSS_QUES2;
		else
			msp->mv_csstate = CSS_START;
		break;
	case CSS_QUES2:
		switch(c)
		{
		case 'h':
			msp->mf_status &= ~MFS_NOMARGIN;  /* auto margin ON */
			break;
		case 'l':
			msp->mf_status |= MFS_NOMARGIN;  /* auto margin OFF */
			break;
		}
		msp->mv_csstate = CSS_START;
		break;
	}
}


/*
 * SCO sequences.  In the "ESC [ =" range
 */

ansi_sco(msp, c)
struct screen *msp;
int c;
{
	int tmp, i;
	int (*adapctl)()=msp->mv_adapter->v_adapctl;

	switch(c)
	{
	case '0':
	case '1':
	case '2':
	case '3':
	case '4':
	case '5':
	case '6':
	case '7':
	case '8':
	case '9':
		i = msp->mv_csindex;
		msp->mv_csparam[i] *= 10;
		msp->mv_csparam[i] += c - '0';
		return;

	case ';':
		if (++msp->mv_csindex >= NCSPARAMS)
			msp->mv_csstate = CSS_START;
		return;

	case 'A':		/* Set Overscan Color */
		(*adapctl)(msp, AC_SETOS, msp->mv_csparam[0]);

		break;
		;

	case 'B':		/* Set Bell Parameters */
		msp->mb_freq = msp->mv_csparam[0];
		msp->mb_time = msp->mv_csparam[1];
		break;

	case 'C':		/* Set Cursor Type */

		tmp = (msp->mv_csparam[0] << 8) + msp->mv_csparam[1];
		(*adapctl)(msp, AC_DEFCSR, tmp);
		break;

	case 'D':		/* Bold Background on/off */
		tmp= (msp->mv_csindex>0) && (msp->mv_csparam[0] != 0);
		(*adapctl)(msp, AC_BOLDBKG, tmp);
		break;

	case 'E':		/* Attribute bit 8 is either blink or bkgnd intensity */
		tmp= (msp->mv_csindex>0) && (msp->mv_csparam[0] != 0);
		(*adapctl)(msp, AC_BLINKB, tmp);
		break;

	case 'F':		/* set Normal Foreground colors */

		(*adapctl)(msp, AC_DEFNF,msp->mv_csparam[0]);
		if ((msp->mf_status & MFS_TMODES) == 0)		/* normal mode  S005 */
			(*adapctl)(msp, AC_ONN, 0);
		break;

	case 'G':		/* set Normal BackGround colors */
		(*adapctl)(msp, AC_DEFNB,msp->mv_csparam[0]);
		if ((msp->mf_status & MFS_TMODES) == 0)		/* normal mode	S005 */
			(*adapctl)(msp, AC_ONN, 0);
		break;

	case 'H':		/* set Reverse Foreground colors */
		(*adapctl)(msp, AC_DEFRF, msp->mv_csparam[0]);
		if ((msp->mf_status & MFS_TMODES) == MFS_REVSTEXT)  /* reverse mode */
			(*adapctl)(msp, AC_ONR, 0);
		break;

	case 'I':		/* set Reverse BackGround colors */
		(*adapctl)(msp, AC_DEFRB, msp->mv_csparam[0]);
		if ((msp->mf_status & MFS_TMODES) == MFS_REVSTEXT)  /* reverse mode */
			(*adapctl)(msp, AC_ONR, 0);
		break;

	case 'J':		/* set Graphic Foreground colors */
		(*adapctl)(msp, AC_DEFGF, msp->mv_csparam[0]);
		if ((msp->mf_status & MFS_TMODES) == MFS_GRAFTEXT)  /* graphics text */
			(*adapctl)(msp, AC_ONG, 0);
		break;

	case 'K':		/* set Graphic BackGround colors */
		(*adapctl)(msp, AC_DEFGB, msp->mv_csparam[0]);
		if ((msp->mf_status & MFS_TMODES) == MFS_GRAFTEXT)  /* graphics text */
			(*adapctl)(msp, AC_ONG, 0);
		break;
	}
	msp->mv_csstate = CSS_START;
}



int ansirange(n, max)
register n;
{
	if (n <= 0)
		n = 1;
	if (n > max)
		n = max;
	return(n);
}
