package Zim::Formats::Html;

use strict;
use Encode;
use Zim::Template;

our $VERSION = '0.17';

=head1 NAME

Zim::Formats::Html - Html dumper for zim

=head1 DESCRIPTION

This is a dumper that can be used to export zim wiki pages to html.

=head1 METHODS

=over 4

=item C<save_tree(IO, TREE, PAGE)>

Converts a parse tree into plain text.

=cut

our %entities = (
	'&' => 'apos',
	'<' => 'lt',
	'>' => 'gt',
);
	
sub save_tree {
	my ($class, $io, $tree, $page) = @_;

	my $template = $page->{repository}{_template};
	$template = length($template)
		? Zim::Template->new($template)
		: Zim::Template->new(\'[% page.body %]')
		unless ref $template ;

	my ($title, $heading) = ($page->basename, '');
	if (ref $$tree[2] and $$tree[2][0] =~ /^head/) {
		# FIXME This logic belongs somewhere central
		$title = $$tree[2];
		$$tree[2] = '';
		$title = join '', @$title[2 .. $#$title];
		$heading = $title;
	}
	# else above defaults

	my @path = $page->namespaces;
	my $root = '../' x scalar @path || './';
	$page->{_root_dir} = $root;

	my $data = {
		page => {
			name => '',
			namespace => '',
			title => $title,
			heading => $heading,
			links => sub {
				[ map { {name => $_, file => _href($page, $_)} }
					sort $page->_list_links ]
			},
			backlinks => sub {
				return [ map { {name => $_, file => _href($page, $_)} }
					sort $page->_list_backlinks ] ;
			},
		},
		prev => {},
		next => {},
		repository => { root => $root },
		zim => { version => $VERSION },
	};
	$data->{page}{body} = sub {
		my (undef, $fh) = @_;
		return unless $fh;
		my $old_fh = select $fh;
		eval {
			$page->{_mode} = lc $data->{mode};
			print qq#<div class="slide">\n# if $page->{_mode} eq 's5';
			_dump_node($tree, $page);
			print qq#</div>\n# if $page->{_mode} eq 's5';
		};
		select $old_fh;
		die $@ if $@;
		return undef; # no caching here
	};
	if ($page->{repository}{_prev}) {
		my $prev = $page->{repository}{_prev}->name;
		$data->{prev} = {
			name => $prev,
			file => _href($page, $prev),
		};
	}
	if ($page->{repository}{_next}) {
		my $next = $page->{repository}{_next};
		$data->{next} = {
			name => $next,
			file => _href($page, $next),
		};
	}

	$template->process($data => $io);
}

sub _dump_node {
	no warnings;
	
	my ($node, $page) = @_;
	my ($name, $opt) = splice @$node, 0, 2;
	
	my @tags;
	if ($name =~ /^head(\d+)/) { @tags = ("<h$1>", "</h$1>\n") }
	elsif ($name eq 'Para') {
		@tags = ("<p>\n", "\n</p>\n");
		if (! ref $$node[0] and $$node[0] =~ /^\s*\*\s+/m) {
			# ugly hack ... should not be necessary
			parse_list($page, @$node);
			@$node = ();
		}
		elsif ( $page->{_mode} eq 's5'
				and ! ref $$node[0]
				and $$node[0] =~ s/^----+\s+//
		) {
			print qq#</div>\n\n\n<div class="slide">\n#;
			shift @$node unless $$node[0] =~ /\S/;
		}
	}
	elsif ($name eq 'Verbatim' ) {  @tags = ("<pre>\n", "</pre>\n") }
	elsif ($name eq 'bold'     ) { @tags = ('<b>', '</b>')   }
	elsif ($name eq 'italic'   ) { @tags = ('<i>', '</i>')   }
	elsif ($name eq 'underline') { @tags = ('<u>', '</u>')   }
	elsif ($name eq 'strike'   ) { @tags = ('<strike>', '</strike>') }
	elsif ($name eq 'verbatim' ) { @tags = ('<tt>', '</tt>') }
	elsif ($name eq 'link') {
		my ($type, $href) = $page->parse_link($$opt{to});
		if ($type eq 'page') {
			my $link = $page->resolve_link($$opt{to});
			$href = _href($page, $link->name) if $link;
		}
		print STDERR "  link: $$opt{to} => $href\n";
		@tags = ("<a href='$href'>", "</a>");
	}
	elsif ($name eq 'image') {
		@tags = ("<img src='$$opt{src}' alt='$$opt{src}' />", '');
	}
	# else recurs silently

	print $tags[0];
	for (@$node) {
		if (ref $_) { _dump_node($_, $page) } # recurse
		else {
			s#([<>&])#\&$entities{$1}\;#g; # encode
			s#\n#<br />\n#g unless $tags[0] =~ m#<pre>#;
			print $_;
		}
	}
	print $tags[1];
}

sub _href {
	my ($page, $href) = @_;
	return '' unless length $href;
	my $namespace = $page->namespace;
	#print STDERR "page: $href, relative: $namespace => ";
	$href =~ s#^:*#:#;
	unless ($href =~ s#^($namespace):*##) {
		$href =~ s/^:*//;
		$href = $page->{_root_dir}.$href;
	}
	$href =~ s#:+$##;
	$href =~ s#[/:]+#/#g;
	$href .= '.html';
	#print STDERR $href."\n";
	$href = Encode::encode_utf8($href); # turn utf8 flag off
	$href =~ s#([^a-zA-Z0-9/\.])#sprintf '%%%02x', ord $1#eg; # url encoding
	return $href;
}

sub parse_list { # FIXME totally wrong level for this kind of parsing :(
	no warnings;

	my ($page, @nodes) = @_;
	
	print "<ul>\n";
	my ($level, $close);
	for (@nodes) {
		if (ref $_) { _dump_node($_, $page) }
		else {
			s#([<>&])#\&$entities{$1}\;#g; # encode
			my @lines = split /[\n\r]+/, $_;
			for (@lines) {
				if ($_ =~ s/^(\s*)\*\s+//) {
					print $close;
					my $lvl = length $1;
					if ($lvl > $level) {
						print "<ul>\n" for 1 .. $lvl - $level;
					}
					elsif ($lvl < $level) {
						print "</ul>\n" for 1 .. $level - $lvl;
					}
					$level = $lvl;
					print "<li>".$_;
					$close = "</li>\n";
				}
				else { print $_ }
			}
		}
	}
	print $close;
	print "</ul>\n" for 1 .. $level + 1;
}

1;

__END__

=back

=head1 AUTHOR

Jaap Karssenberg (Pardus) E<lt>pardus@cpan.orgE<gt>

Copyright (c) 2005 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 SEE ALSO

=cut

