/**********************************************************************
zyGrib: meteorological GRIB file viewer
Copyright (C) 2008-2010 - Jacques Zaninetti - http://www.zygrib.org

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
***********************************************************************/

#ifndef MBLUERECORD_H
#define MBLUERECORD_H

#include "IrregularGridded.h"
#include "zuFile.h"

// Line format:
// ID,
// 2m temp (C),
// 2m RH (%),
// total precip (mm),
// 10m u (m/s),
// 10m v (m/s),
// total clouds (%),
// wind gusts (m/s),
// p at sea level (Pa),
// surface temp (C),
// rain probability 0.1mm/h (%)

enum MblueDataIndexes {
		MB_temp2m=0, 
		MB_rh2m, 
		MB_precip,
		MB_windu10m, 
		MB_windv10m,
		MB_clouds, 
		MB_gust, 
		MB_pmsl, 
		MB_tempsfc, 
		MB_rainprob,
		MB_dewpoint2m,
		
		MB_NBVALUES		// don't remove
};

//------------------------------------------------------
class MbluePoint
{
	public:
		MbluePoint ();
		MbluePoint (ZUFILE *file, int nbdata);
		
		// Interpolate values from other points
		MbluePoint ( double x, double y,
					 int  nbPoints,
					 MbluePoint *p1,
					 MbluePoint *p2,
					 MbluePoint *p3,
					 MbluePoint *p4 );

		double getValue (DataCode dtc);
		double getValue (int index);
		
		float x, y;
		bool isOk() 	{return ok;}

		static int getDataIndex (DataCode dtc);
		
	private:
		int  nbvalues;
		bool ok;
		
		float  values [MB_NBVALUES];
		bool   read_float32 (ZUFILE *file, float *val);

};

//===================================================
class MblueRecord : public IrregularGridRecord
{
	public:
		MblueRecord (ZUFILE *file);
		~MblueRecord ();

		bool isOk() const	{return ok;}
		int  findNeighbour_clusters (
					double lon, double lat,
					MbluePoint **p1, MbluePoint **p2, 
					MbluePoint **p3, MbluePoint **p4 ) const;
		
        virtual double  getInterpolatedValue (
						DataCode dtc,
						double px, double py,
						bool interpolateValues=true ) const;

		bool getZoneExtension (double *x0,double *y0, double *x1,double *y1);
		
        time_t getRecordRefDate () const        { return refDate; }
        time_t getRecordCurrentDate () const    { return curDate; }
	
		std::list <MbluePoint *> * getListPoints () {return &allPoints;}
		
		/** All records have (or simulate) a rectangular regular grid.
		*/ 
		virtual double getValueOnRegularGrid (DataCode dtc, int i, int j) const;
		double getSmoothPressureMSL (int i, int j) const
					{ return smoothPressureGrid [i+j*Ni]; }
		
		virtual int     getNi () const;
        virtual int     getNj () const;
        virtual double  getX (int i) const;
        virtual double  getY (int j) const;
        virtual double  getDeltaX () const;
        virtual double  getDeltaY () const;
		
        virtual int    getTotalNumberOfPoints ()  const
						{ return ok ? allPoints.size() : 0; }
        virtual double getAveragePointsDensity () const
						{ return ok ? allPoints.size()/((xmax-xmin)*(ymax-ymin)) : 0; }

        virtual int  getIdCenter() const { return 0; }
        virtual int  getIdModel()  const { return 0; }
        virtual int  getIdGrid()   const { return 0; }

	private:
		bool ok;
		std::list <MbluePoint *> allPoints;
        time_t refDate;      // Reference date
        time_t curDate;      // Current date

		std::list <MbluePoint *> *clusters;	// table of clusters of points
		int clustersNBX, clustersNBY;	// size of the grid of clusters
		void  	makeClusters();
		
		MbluePoint **regularGrid;    // Virtual regular grid parameters
		int    Ni, Nj;	
		double Di, Dj;
		void   makeVirtualRegularGrid ();
		
		float  *smoothPressureGrid;	    // try to reduce pressure noise
		void   makeSmoothPressureGrid ();
		
		std::list <MbluePoint *> & getCluster (
										double x, double y) const;
		std::list <std::list <MbluePoint *> > *getClustersList (
										double x, double y) const;

		bool   readFile (ZUFILE *file);
		bool   read_int32   (ZUFILE *file, int *val);
		
		int   findBestNeighboursInCluster (
						double lon, double lat,
						std::list <MbluePoint *> cluster,
						MbluePoint **p1, MbluePoint **p2, 
						MbluePoint **p3, MbluePoint **p4, 
						double *d1, double *d2, double *d3, double *d4
					) const;

		double  getInterpolatedValue (
					 int index, 
					 double px, double py,
					 bool interpolateValues=true ) const;
		virtual double getValueOnRegularGrid (int ind, int i, int j) const;
};


 

#endif
