/* Copyright (c) 2002, 2002 Michael J Roberts.  All Rights Reserved. */
/*
Name
  tclibprs.cpp - tads compiler: library parser
Function
  Parses .tl files, which are text files that reference source files
  for inclusion in a compilation.  A .tl file can be used in a compilation
  as though it were a source file, and stands for the set of source files
  it references.  A .tl file can reference other .tl files.
Notes
  
Modified
  01/08/02 MJRoberts  - Creation
*/

#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include "os.h"
#include "t3std.h"
#include "tclibprs.h"


/*
 *   Instantiate.  We'll note the name of the library, but we don't attempt
 *   to open it at this point - we won't open the file until parse_lib() is
 *   called.  
 */
CTcLibParser::CTcLibParser(const char *lib_name)
{
    char lib_path[OSFNMAX];

    /* 
     *   Extract the library's directory path.  Since files within a library
     *   are always relative to the directory containing the library itself,
     *   we use the full path to the library as the starting point for each
     *   file found within the library.  
     */
    os_get_path_name(lib_path, sizeof(lib_path), lib_name);

    /* remember the library path */
    lib_path_ = lib_copy_str(lib_path);

    /* remember the name of the library */
    lib_name_ = lib_copy_str(lib_name);

    /* no errors yet */
    err_cnt_ = 0;

    /* no lines read yet */
    linenum_ = 0;
}

CTcLibParser::~CTcLibParser()
{
    /* delete our library filename and path */
    lib_free_str(lib_name_);
    lib_free_str(lib_path_);
}

/*
 *   Parse a library file. 
 */
void CTcLibParser::parse_lib()
{
    osfildef *fp;

    /* open our file - if it fails, flag the error and terminate */
    fp = osfoprt(lib_name_, OSFTTEXT);
    if (fp == 0)
    {
        err_open_file();
        ++err_cnt_;
        return;
    }

    /* keep going until we run out of file */
    for (;;)
    {
        char buf[1024];
        size_t len;
        char *p;
        char *sep;
        char *var_name;
        char *var_val;

        /* read the next line of text */
        if (osfgets(buf, sizeof(buf), fp) == 0)
            break;

        /* count the line */
        ++linenum_;

        /* check for a terminating newline sequence */
        len = strlen(buf);
        while (len != 0 && (buf[len-1] == '\n' || buf[len-1] == '\r'))
            --len;

        /* if we didn't find a newline, the line is too long */
        if (buf[len] == '\0')
        {
            char buf2[128];
            
            /* 
             *   If we can read anything more from the file, this indicates
             *   that the line was simply too long to read into our buffer;
             *   skip the rest of the line and log an error in this case.
             *   If we can't read any more text, it means we've reached the
             *   end of the file. 
             */
            if (osfgets(buf2, sizeof(buf2), fp) != 0)
            {
                /* there's more, so the line is too long - flag an error */
                err_line_too_long();
                ++err_cnt_;

                /* skip text until we find a newline */
                for (;;)
                {
                    /* if we've found our newline, we're done skipping */
                    len = strlen(buf2);
                    if (len != 0
                        && (buf2[len-1] == '\n' || buf2[len-1] == '\r'))
                        break;

                    /* read another chunk */
                    if (osfgets(buf2, sizeof(buf2), fp) == 0)
                        break;
                }

                /* we've skipped the line, so go back for the next */
                continue;
            }
        }

        /* remove the newline from the buffer */
        buf[len] = '\0';

        /* skip leading spaces */
        for (p = buf ; isspace(*p) ; ++p) ;

        /* if the line is empty or starts with '#', skip it */
        if (*p == '\0' || *p == '#')
            continue;

        /* the variable name starts here */
        var_name = p;

        /* the variable name ends at the next space or colon */
        for ( ; *p != '\0' && *p != ':' && !isspace(*p) ; ++p) ;

        /* note where the variable name ends */
        sep = p;

        /* skip spaces after the end of the variable name */
        for ( ; isspace(*p) ; ++p) ;

        /* 
         *   if we didn't find a colon after the variable name, it's an
         *   error; flag the error and skip the line 
         */
        if (*p != ':')
        {
            err_missing_colon();
            ++err_cnt_;
            continue;
        }

        /* put a null terminator at the end of the variable name */
        *sep = '\0';

        /* skip any spaces after the colon */
        for (++p ; isspace(*p) ; ++p) ;

        /* the value starts here */
        var_val = p;

        /* scan the value */
        scan_var(var_name, var_val);
    }

    /* we're done - close the file */
    osfcls(fp);
}

/*
 *   Scan a variable 
 */
void CTcLibParser::scan_var(const char *name, const char *val)
{
    /* call the appropriate routine based on the variable name */
    if (stricmp(name, "name") == 0)
        scan_name(val);
    else if (stricmp(name, "source") == 0)
        scan_source(val);
    else if (stricmp(name, "library") == 0)
        scan_library(val);
    else
        err_unknown_var(name, val);
}

/*
 *   Scan a source filename.  We build the full filename by combining the
 *   library path and the given filename, then we call scan_full_source().  
 */
void CTcLibParser::scan_source(const char *val)
{
    char rel_path[OSFNMAX];
    char full_name[OSFNMAX];

    /* convert the value from a URL-style path to a local path */
    os_cvt_url_dir(rel_path, sizeof(rel_path), val, FALSE);

    /* build the full name */
    os_build_full_path(full_name, sizeof(full_name), lib_path_, rel_path);

    /* call the full filename scanner */
    scan_full_source(val, full_name);
}

/*
 *   Scan a library filename.  We build the full filename by combining the
 *   enclosing library path and the given filename, then we call
 *   scan_full_library().  
 */
void CTcLibParser::scan_library(const char *val)
{
    char rel_path[OSFNMAX];
    char full_name[OSFNMAX];

    /* convert the value from a URL-style path to a local path */
    os_cvt_url_dir(rel_path, sizeof(rel_path), val, FALSE);

    /* build the full name */
    os_build_full_path(full_name, sizeof(full_name), lib_path_, rel_path);

    /* call the full library filename scanner */
    scan_full_library(val, full_name);
}

/* 
 *   log an error in a source line 
 */
void CTcLibParser::src_err_msg(const char *msg, ...)
{
    char buf[1024];
    va_list args;

    /* format the caller's message and its arguments */
    va_start(args, msg);
    vsprintf(buf, msg, args);
    va_end(args);

    /* display the message with the source name and line number */
    err_msg("%s (%lu): %s", lib_name_, linenum_, msg);
}
