/*
 * Copyright 2009-2019 The VOTCA Development Team (http://www.votca.org)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
#define BOOST_TEST_MAIN

#define BOOST_TEST_MODULE overlap_filter_test
#include <boost/test/unit_test.hpp>
#include <iostream>
#include <votca/xtp/filterfactory.h>

using namespace votca::xtp;

BOOST_AUTO_TEST_SUITE(overlap_filter_test)

BOOST_AUTO_TEST_CASE(coeffs_test) {
  FilterFactory::RegisterAll();
  std::unique_ptr<StateFilter_base> rho_f =
      std::unique_ptr<StateFilter_base>(Filter().Create("overlap"));

  std::ofstream opt("overlap_filter.xml");
  opt << "<overlap>0.0045</overlap>" << std::endl;
  opt.close();
  votca::tools::Property prop;
  prop.LoadFromXML("overlap_filter.xml");
  rho_f->Initialize(prop.get("overlap"));

  std::ofstream xyzfile("molecule.xyz");
  xyzfile << " 5" << std::endl;
  xyzfile << " methane" << std::endl;
  xyzfile << " C            .000000     .000000     .000000" << std::endl;
  xyzfile << " H            .629118     .629118     .629118" << std::endl;
  xyzfile << " H           -.629118    -.629118     .629118" << std::endl;
  xyzfile << " H            .629118    -.629118    -.629118" << std::endl;
  xyzfile << " H           -.629118     .629118    -.629118" << std::endl;
  xyzfile.close();

  std::ofstream basisfile("3-21G.xml");
  basisfile << "<basis name=\"3-21G\">" << std::endl;
  basisfile << "  <element name=\"H\">" << std::endl;
  basisfile << "    <shell scale=\"1.0\" type=\"S\">" << std::endl;
  basisfile << "      <constant decay=\"5.447178e+00\">" << std::endl;
  basisfile << "        <contractions factor=\"1.562850e-01\" type=\"S\"/>"
            << std::endl;
  basisfile << "      </constant>" << std::endl;
  basisfile << "      <constant decay=\"8.245470e-01\">" << std::endl;
  basisfile << "        <contractions factor=\"9.046910e-01\" type=\"S\"/>"
            << std::endl;
  basisfile << "      </constant>" << std::endl;
  basisfile << "    </shell>" << std::endl;
  basisfile << "    <shell scale=\"1.0\" type=\"S\">" << std::endl;
  basisfile << "      <constant decay=\"1.831920e-01\">" << std::endl;
  basisfile << "        <contractions factor=\"1.000000e+00\" type=\"S\"/>"
            << std::endl;
  basisfile << "      </constant>" << std::endl;
  basisfile << "    </shell>" << std::endl;
  basisfile << "  </element>" << std::endl;
  basisfile << "  <element name=\"C\">" << std::endl;
  basisfile << "    <shell scale=\"1.0\" type=\"S\">" << std::endl;
  basisfile << "      <constant decay=\"1.722560e+02\">" << std::endl;
  basisfile << "        <contractions factor=\"6.176690e-02\" type=\"S\"/>"
            << std::endl;
  basisfile << "      </constant>" << std::endl;
  basisfile << "      <constant decay=\"2.591090e+01\">" << std::endl;
  basisfile << "        <contractions factor=\"3.587940e-01\" type=\"S\"/>"
            << std::endl;
  basisfile << "      </constant>" << std::endl;
  basisfile << "      <constant decay=\"5.533350e+00\">" << std::endl;
  basisfile << "        <contractions factor=\"7.007130e-01\" type=\"S\"/>"
            << std::endl;
  basisfile << "      </constant>" << std::endl;
  basisfile << "    </shell>" << std::endl;
  basisfile << "    <shell scale=\"1.0\" type=\"SP\">" << std::endl;
  basisfile << "      <constant decay=\"3.664980e+00\">" << std::endl;
  basisfile << "        <contractions factor=\"-3.958970e-01\" type=\"S\"/>"
            << std::endl;
  basisfile << "        <contractions factor=\"2.364600e-01\" type=\"P\"/>"
            << std::endl;
  basisfile << "      </constant>" << std::endl;
  basisfile << "      <constant decay=\"7.705450e-01\">" << std::endl;
  basisfile << "        <contractions factor=\"1.215840e+00\" type=\"S\"/>"
            << std::endl;
  basisfile << "        <contractions factor=\"8.606190e-01\" type=\"P\"/>"
            << std::endl;
  basisfile << "      </constant>" << std::endl;
  basisfile << "    </shell>" << std::endl;
  basisfile << "    <shell scale=\"1.0\" type=\"SP\">" << std::endl;
  basisfile << "      <constant decay=\"1.958570e-01\">" << std::endl;
  basisfile << "        <contractions factor=\"1.000000e+00\" type=\"S\"/>"
            << std::endl;
  basisfile << "        <contractions factor=\"1.000000e+00\" type=\"P\"/>"
            << std::endl;
  basisfile << "      </constant>" << std::endl;
  basisfile << "    </shell> " << std::endl;
  basisfile << "  </element>" << std::endl;
  basisfile << "</basis>" << std::endl;
  basisfile.close();

  Orbitals A;
  A.setDFTbasisName("3-21G.xml");
  A.QMAtoms().LoadFromFile("molecule.xyz");
  A.setBasisSetSize(17);
  A.setNumberOfAlphaElectrons(5);
  A.setNumberOfOccupiedLevels(5);
  A.MOs().eigenvalues() = Eigen::VectorXd::Zero(17);
  A.MOs().eigenvalues() << -19.8117, -6.22408, -6.14094, -6.14094, -6.14094,
      -3.72889, -3.72889, -3.72889, -3.64731, -3.09048, -3.09048, -3.09048,
      -2.63214, -2.08206, -2.08206, -2.08206, -2.03268;

  A.QPpertEnergies() = Eigen::VectorXd::Zero(17);
  A.QPpertEnergies() << -10.189, -1.01045, -0.620145, -0.620146, -0.620148,
      0.261183, 0.348342, 0.348343, 0.348342, 0.920829, 0.920829, 0.920829,
      1.18002, 1.27325, 1.27325, 1.27325, 1.96983;

  A.MOs().eigenvectors() = Eigen::MatrixXd::Zero(17, 17);
  A.MOs().eigenvectors() << -0.996559, -0.223082, 4.81443e-15, 2.21045e-15,
      -6.16146e-17, -3.16966e-16, 5.46703e-18, -1.09681e-15, -0.0301914,
      6.45993e-16, 1.05377e-16, 3.41154e-16, -0.102052, -5.79826e-16,
      9.38593e-16, -4.69346e-15, -0.111923, -0.0445146, 0.88316, -1.94941e-14,
      -8.67388e-15, -7.26679e-16, 1.16326e-14, -3.35886e-15, 2.37877e-14,
      0.866126, 3.2068e-15, 3.80914e-15, 3.24563e-15, -0.938329, -6.4404e-15,
      1.10811e-14, -5.5056e-14, -1.28767, 8.15798e-17, 2.30849e-14, 1.04169,
      0.117804, 0.0951759, 0.179467, 0.147031, 0.39183, -1.02927e-14, 0.32699,
      -0.223689, -0.130009, 1.0375e-15, -0.0940179, 0.126956, 0.0122904,
      1.41709e-15, 4.60157e-17, -7.1203e-15, 0.143338, -0.980459, -0.355251,
      0.41611, -0.10826, -0.149964, 2.41546e-16, 0.12214, -0.0512447, 0.39537,
      1.1054e-15, -0.0996828, -0.0636092, -0.105478, 5.10746e-15, -5.25872e-18,
      4.8424e-15, 0.0488925, 0.364515, -0.9863, 0.0447336, 0.417155, -0.177023,
      5.76117e-15, -0.228081, -0.348136, 0.0253377, -1.05286e-15, 0.079576,
      0.0703157, -0.117608, 5.31327e-15, 0.0472716, 0.235837, -3.58018e-15,
      -1.68354e-15, 2.3989e-15, -9.86879e-15, 4.52519e-15, -1.6106e-14,
      -0.599523, -1.31237e-14, -8.63443e-15, -8.61196e-15, 1.8333, 2.05275e-14,
      -3.9562e-14, 1.89874e-13, 4.24316, -2.74184e-16, -1.53939e-15, -0.162416,
      -0.0183675, -0.0148395, -0.151162, -0.123842, -0.330032, 1.10084e-15,
      -1.45092, 0.992556, 0.576875, -3.82954e-15, 0.604373, -0.816111,
      -0.0790061, -8.89474e-15, -2.24862e-16, 3.23655e-15, -0.0223487, 0.152869,
      0.0553894, -0.350483, 0.0911859, 0.126313, -5.48468e-15, -0.541961,
      0.227383, -1.75434, -3.89443e-15, 0.640788, 0.408897, 0.67804,
      -3.17156e-14, -2.81346e-17, -1.09423e-15, -0.00762313, -0.0568338,
      0.15378, -0.0376785, -0.351364, 0.149104, -4.94425e-15, 1.01204, 1.54475,
      -0.112429, 8.50653e-15, -0.511536, -0.452008, 0.756019, -3.3547e-14,
      -0.00106227, 0.0237672, 0.00345981, -0.00139675, -0.00349474, -0.597906,
      -0.425733, -0.0605479, -0.343823, 0.162103, -0.45692, 0.21318, -0.600309,
      0.310843, -0.36406, 0.574148, 0.0554949, -0.00928842, -0.0414346,
      0.0619999, -0.0250297, -0.0626259, 0.00227746, 0.00162164, 0.00023063,
      -0.0301047, 0.273177, -0.770004, 0.359253, 0.0095153, -0.8783, 1.02867,
      -1.62228, -1.24406, -0.00106227, 0.0237672, 0.00238182, 0.00205737,
      0.00402848, 0.262742, 0.151145, -0.671213, -0.343823, 0.317484, 0.12884,
      -0.40386, -0.600309, 0.201313, -0.327527, -0.641099, 0.0554949,
      -0.00928842, -0.0414346, 0.0426822, 0.0368682, 0.0721904, -0.0010008,
      -0.000575719, 0.00255669, -0.0301047, 0.535026, 0.217123, -0.680588,
      0.0095153, -0.568818, 0.925441, 1.81145, -1.24406, -0.00106227, 0.0237672,
      -0.00318563, 0.0034409, -0.00203628, 0.514364, -0.353326, 0.391148,
      -0.343823, -0.496623, -0.0536813, -0.176018, -0.600309, -0.744328,
      -0.01898, 0.0665156, 0.0554949, -0.00928842, -0.0414346, -0.0570866,
      0.0616609, -0.0364902, -0.00195924, 0.00134584, -0.0014899, -0.0301047,
      -0.836913, -0.0904642, -0.296627, 0.0095153, 2.10313, 0.0536287,
      -0.187943, -1.24406, -0.00106227, 0.0237672, -0.002656, -0.00410152,
      0.00150255, -0.1792, 0.627913, 0.340613, -0.343823, 0.0170366, 0.38176,
      0.366698, -0.600309, 0.232172, 0.710567, 0.000435528, 0.0554949,
      -0.00928842, -0.0414346, -0.0475955, -0.0734994, 0.0269257, 0.000682583,
      -0.00239176, -0.00129742, -0.0301047, 0.0287103, 0.643346, 0.617962,
      0.0095153, -0.656011, -2.00774, -0.0012306, -1.24406;

  A.setBSEindices(0, 16);
  A.setTDAApprox(true);
  A.setGWindices(0, 16);
  Eigen::MatrixXd spsi_ref = Eigen::MatrixXd::Zero(60, 3);
  spsi_ref << -0.00178316, -0.0558332, 0.0151767, 0.00568291, 0.0149417,
      0.0556358, 0.05758, -0.00320371, -0.00502105, 0.00379328, -0.00232255,
      -0.00817518, -0.00848959, -0.000618633, -0.00376334, -0.000395809,
      -0.00899117, 0.0023708, 7.13955e-08, 3.47498e-08, -1.08537e-08,
      0.000420413, 0.011896, -0.00320024, -0.00288487, 0.00320821, 0.0115465,
      0.0119767, 0.000355172, 0.00289343, -2.55565e-08, 1.91684e-08,
      3.01647e-09, 6.84051e-09, 2.79592e-10, -1.35767e-09, 0.00420618, 0.092451,
      -0.0239374, 0.0036276, 0.0113951, 0.0471896, 0.0465325, -0.00398807,
      -0.00484251, 0.0051614, -0.0031325, -0.0112447, -0.010968, -0.000896935,
      -0.00488789, 0.000951266, 0.0239709, -0.00630323, 0.000419006, 0.0201651,
      -0.00573095, -0.00118124, -0.0269275, 0.00700955, -0.00345217, 0.00356488,
      0.0134361, 0.013156, 9.58532e-05, 0.00315613, -2.58268e-05, -0.00124098,
      0.000352706, -1.80679e-06, -8.71959e-05, 2.47799e-05, -0.0150073,
      0.0114186, 0.0443012, 0.0180327, -0.026287, 0.0734351, -0.0643994,
      0.0257628, 0.0132084, -0.0123339, 0.0092297, -0.0148779, 0.0122493,
      -0.00246837, -0.0125735, -0.00375172, 0.00294872, 0.0112899, 0.00648758,
      -0.0055755, -0.0191436, 0.00433063, -0.00332619, -0.0128321, 0.0111166,
      -0.00969272, 0.0189659, -0.0160119, 0.00458659, 0.0107104, -0.000399315,
      0.000343129, 0.00117813, -2.80525e-05, 2.41086e-05, 8.2778e-05,
      -0.0450479, -0.00034974, -0.015063, 0.0655838, 0.0115163, -0.022358,
      0.0220978, 0.0583236, 0.0513097, -0.0119156, 0.00490159, 0.0116429,
      -0.0132479, -0.0146227, -0.00863565, -0.0118978, -0.000282044,
      -0.00400272, 0.0199347, 0.00139057, 0.00635067, 0.0131991, 0.000163177,
      0.00441453, 0.0159091, -0.00241207, -0.0110696, 0.0123057, 0.0171503,
      0.0119626, -0.00122682, -8.55716e-05, -0.00039083, -8.62007e-05,
      -6.0128e-06, -2.746e-05, -0.0304688, -0.954049, 0.259333, 0.0971056,
      0.255313, 0.950672, 0.983887, -0.0547431, -0.0857965, 0.0170489,
      -0.0104387, -0.036743, -0.0381557, -0.00278036, -0.0169143, -0.00177889,
      -0.04041, 0.0106552, -2.23782e-07, 2.40738e-07, 1.03401e-07, -0.000182535,
      -0.00516415, 0.00138942, 0.00125201, -0.00139237, -0.00501195,
      -0.00519809, -0.000154171, -0.00125602, 4.03664e-08, -6.04796e-08,
      -4.6768e-08, -2.38233e-09, 2.31605e-09, 1.35922e-09;

  A.BSESinglets().eigenvectors() = spsi_ref;

  // reference energy
  Eigen::VectorXd se_ref = Eigen::VectorXd::Zero(3);
  se_ref << 0.107455, 0.107455, 0.107455;

  A.BSESinglets().eigenvalues() = se_ref;
  A.CalcCoupledTransition_Dipoles();

  BOOST_CHECK_EQUAL(rho_f->NeedsInitialState(), true);

  rho_f->UpdateHist(A, QMState("s1"));

  std::vector<votca::Index> ref = {0};
  std::vector<votca::Index> results =
      rho_f->CalcIndeces(A, QMStateType::Singlet);

  BOOST_CHECK_EQUAL(results.size(), ref.size());
  for (votca::Index i = 0; i < votca::Index(ref.size()); i++) {
    BOOST_CHECK_EQUAL(ref[i], results[i]);
  }

  rho_f->UpdateHist(A, QMState("s2"));
  std::vector<votca::Index> results2 =
      rho_f->CalcIndeces(A, QMStateType::Singlet);
  std::vector<votca::Index> ref2 = {1};
  for (votca::Index i = 0; i < votca::Index(ref2.size()); i++) {
    BOOST_CHECK_EQUAL(ref2[i], results2[i]);
  }
  BOOST_CHECK_EQUAL(results2.size(), ref2.size());
  std::unique_ptr<StateFilter_base> rho_f2 =
      std::unique_ptr<StateFilter_base>(Filter().Create("overlap"));

  rho_f2->Initialize(prop.get("overlap"));

  rho_f2->UpdateHist(A, QMState("pqp8"));
  std::vector<votca::Index> results3 =
      rho_f2->CalcIndeces(A, QMStateType::PQPstate);

  std::vector<votca::Index> ref3 = {8};
  BOOST_CHECK_EQUAL(results3.size(), ref3.size());
  for (votca::Index i = 0; i < votca::Index(ref3.size()); i++) {
    BOOST_CHECK_EQUAL(ref3[i], results3[i]);
  }

  // reference energy
  Eigen::VectorXd se_ref_btda = Eigen::VectorXd::Zero(3);
  se_ref_btda << 0.0887758, 0.0887758, 0.0887758;

  // reference coefficients
  Eigen::MatrixXd spsi_ref_btda = Eigen::MatrixXd::Zero(60, 3);
  spsi_ref_btda << 0.00228773, -0.0572672, 0.0185901, -0.00606479, 0.0182899,
      0.0570883, -0.0599032, -0.00403878, -0.00506982, -0.00239507, -0.00174279,
      -0.00511526, 0.005396, -0.000467089, -0.0023674, 0.000293773, -0.00562893,
      0.00178028, -1.18994e-08, 3.6883e-08, 1.39133e-09, -0.00038655,
      0.00890738, -0.00286588, 0.00216846, 0.00287559, 0.00864496, -0.00910256,
      0.000306734, 0.00218106, -6.67006e-09, 1.50464e-08, -4.23043e-09,
      5.73874e-11, 1.02039e-09, 3.42402e-10, -0.00300256, 0.0559254, -0.0174141,
      -0.00229461, 0.00836668, 0.0285808, -0.028529, -0.00280458, -0.00285455,
      -0.00278921, -0.00201767, -0.00602331, 0.00596683, -0.000570189,
      -0.00262914, -0.00061392, 0.0128447, -0.00405267, -0.000330257, 0.011532,
      -0.00389023, 0.000903715, -0.0173665, 0.00543237, 0.00223576, 0.00277158,
      0.0086669, -0.00860769, 9.71086e-05, 0.00205762, 5.28565e-05, -0.00184564,
      0.000622629, 6.78691e-07, -2.36991e-05, 7.99472e-06, 0.00913939,
      0.00838224, 0.0268121, -0.0109663, -0.0140137, 0.0457817, 0.0393868,
      0.01648, 0.00733715, 0.0066705, 0.00467073, -0.00831923, -0.0066442,
      -0.00172131, -0.00676294, 0.00202369, 0.00191325, 0.00605252, -0.00373624,
      -0.00379982, -0.0109468, -0.00281175, -0.00260087, -0.00827985,
      -0.00723077, -0.00578823, 0.0126979, 0.0104477, 0.00341532, 0.00685506,
      0.000597973, 0.000608139, 0.00175196, 7.67794e-06, 7.80883e-06,
      2.24965e-05, 0.0276451, -0.000451369, -0.00921566, -0.0402838, 0.00608812,
      -0.014051, -0.0138464, 0.0371613, 0.0297107, 0.00645305, 0.00301651,
      0.00619365, 0.00726364, -0.0081113, -0.00430282, 0.00646942, -0.000209234,
      -0.00216462, -0.0115828, 0.000896897, 0.00364201, -0.00863663,
      0.000180708, 0.00287774, -0.010391, -0.00200537, -0.00715858, -0.00813919,
      0.0115232, 0.00727228, 0.00185377, -0.000143538, -0.000582886,
      2.38027e-05, -1.84326e-06, -7.48457e-06, 0.0391594, -0.980256, 0.318212,
      -0.103812, 0.31307, 0.977192, -1.02537, -0.0691326, -0.0867812,
      -0.00843227, -0.00613583, -0.0180091, 0.0189974, -0.00164438, -0.00833481,
      0.00103433, -0.0198172, 0.00626757, 7.10218e-08, 2.05723e-07, 5.20066e-08,
      0.000176091, -0.00405684, 0.00130541, -0.000987275, -0.00130929,
      -0.00393686, 0.00414458, -0.000139691, -0.000993321, -9.14836e-09,
      -7.65102e-08, -3.81041e-08, -1.73462e-10, 2.76218e-09, 6.98481e-10;

  // // reference coefficients AR
  Eigen::MatrixXd spsi_ref_btda_AR = Eigen::MatrixXd::Zero(60, 3);
  spsi_ref_btda_AR << 0.000584144, -0.0146225, 0.00474677, -0.00154856,
      0.00466997, 0.0145766, -0.0152954, -0.00103125, -0.00129451, 0.0017559,
      0.00127769, 0.00375014, -0.00395598, 0.000342451, 0.00173561,
      -0.000215381, 0.00412671, -0.00130515, -5.02586e-09, 4.52834e-08,
      1.34149e-08, 0.000156138, -0.00359781, 0.00115758, -0.000875892,
      -0.00116148, -0.00349181, 0.00367662, -0.000123883, -0.000880951,
      -5.20546e-09, 2.56561e-08, 6.1721e-09, -1.30377e-10, -3.42705e-10,
      -1.4581e-10, 0.00225763, -0.0420501, 0.0130936, 0.00172531, -0.00629088,
      -0.0214897, 0.0214509, 0.00210875, 0.00214631, 0.00443924, 0.00321126,
      0.00958646, -0.00949663, 0.000907487, 0.00418444, 0.000977068, -0.020443,
      0.00645007, 0.000411481, -0.0143682, 0.00484706, -0.000752112, 0.0144531,
      -0.00452103, -0.00186068, -0.00230664, -0.0072129, 0.00716363,
      -8.0822e-05, -0.00171243, 4.85758e-05, -0.00169622, 0.000572232,
      -4.64045e-06, 0.000162032, -5.46606e-05, -0.00687191, -0.0063025,
      -0.0201599, 0.00824556, 0.0105368, -0.034423, -0.0296148, -0.0123913,
      -0.00551674, -0.0106166, -0.00743383, 0.0132406, 0.0105747, 0.00273958,
      0.0107636, -0.00322083, -0.00304507, -0.00963294, 0.00465513, 0.00473433,
      0.013639, 0.00234006, 0.00216454, 0.00689084, 0.00601773, 0.00481718,
      -0.0105676, -0.00869493, -0.00284236, -0.00570504, 0.000549541,
      0.000558892, 0.0016101, -5.24965e-05, -5.33896e-05, -0.000153809,
      -0.0207863, 0.000339395, 0.0069292, 0.0302893, -0.00457762, 0.0105648,
      0.010411, -0.0279414, -0.0223393, -0.0102705, -0.00480099, -0.00985759,
      -0.0115606, 0.0129097, 0.00684819, -0.0102966, 0.000333027, 0.00344514,
      0.0144314, -0.00111747, -0.00453772, 0.00718776, -0.000150389,
      -0.00239498, 0.00864786, 0.00166895, 0.00595764, 0.00677374, -0.00959,
      -0.00605226, 0.00170364, -0.000131921, -0.000535686, -0.000162745,
      1.2602e-05, 5.11726e-05, 0.00995914, -0.249299, 0.0809272, -0.0264016,
      0.07962, 0.248518, -0.260775, -0.0175818, -0.0220701, 0.0109283,
      0.00795199, 0.0233398, -0.0246207, 0.00213124, 0.0108019, -0.00134058,
      0.0256833, -0.00812285, 1.49164e-08, -4.69903e-09, 5.23125e-09,
      6.14995e-07, -1.40366e-05, 4.50767e-06, -3.26276e-06, -4.40871e-06,
      -1.35086e-05, 1.40497e-05, -4.71623e-07, -3.42445e-06, 1.99993e-09,
      -3.51811e-08, -4.56933e-09, -2.19573e-10, 6.10086e-10, -2.35986e-11;

  A.BSESinglets().eigenvectors() = spsi_ref_btda;
  A.BSESinglets().eigenvectors2() = spsi_ref_btda_AR;

  A.BSESinglets().eigenvalues() = se_ref_btda;
  A.setTDAApprox(false);
  A.CalcCoupledTransition_Dipoles();

  std::unique_ptr<StateFilter_base> rho_f3 =
      std::unique_ptr<StateFilter_base>(Filter().Create("overlap"));

  rho_f3->Initialize(prop.get("overlap"));

  rho_f3->UpdateHist(A, QMState("s1"));

  std::vector<votca::Index> ref_btda = {0};
  std::vector<votca::Index> results_btda =
      rho_f3->CalcIndeces(A, QMStateType::Singlet);

  BOOST_CHECK_EQUAL(results_btda.size(), ref_btda.size());
  for (votca::Index i = 0; i < votca::Index(ref_btda.size()); i++) {
    BOOST_CHECK_EQUAL(ref_btda[i], results_btda[i]);
  }
}

BOOST_AUTO_TEST_SUITE_END()
